<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Vessel;
use App\Models\Certificate;
use App\Models\VesselCertificate;

class FixVesselCertificates extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'vessels:fix-certificates';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fix vessels with 0 certificates by ensuring proper relationships';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting to fix vessels with 0 certificates...');

        // Get all vessels
        $vessels = Vessel::all();
        $this->info("Found {$vessels->count()} vessels");

        // Begin transaction
        DB::beginTransaction();

        try {
            $fixedCount = 0;
            $alreadyOkCount = 0;

            foreach ($vessels as $vessel) {
                $certificateCount = VesselCertificate::where('vessel_id', $vessel->id)->count();
                
                if ($certificateCount === 0) {
                    // This vessel has 0 certificates, let's check if there are any in the database
                    $this->info("Vessel {$vessel->name} (ID: {$vessel->id}) has 0 certificates");
                    
                    // Check if there are any certificates in the SQL file for this vessel
                    $sqlCertificates = $this->getCertificatesFromSql($vessel->id);
                    
                    if (count($sqlCertificates) > 0) {
                        $this->info("Found {$sqlCertificates} certificates in SQL file for vessel {$vessel->name}");
                        
                        // Import certificates from SQL file
                        $importedCount = $this->importCertificatesFromSql($vessel->id, $sqlCertificates);
                        
                        if ($importedCount > 0) {
                            $this->info("Imported {$importedCount} certificates for vessel {$vessel->name}");
                            $fixedCount++;
                        }
                    } else {
                        $this->warn("No certificates found in SQL file for vessel {$vessel->name}");
                    }
                } else {
                    $this->line("Vessel {$vessel->name} (ID: {$vessel->id}) already has {$certificateCount} certificates");
                    $alreadyOkCount++;
                }
            }

            // Commit transaction
            DB::commit();
            
            $this->info("Fix completed successfully!");
            $this->info("Fixed {$fixedCount} vessels with 0 certificates");
            $this->info("{$alreadyOkCount} vessels already had certificates");
            
            return 0;
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();
            
            $this->error("Error during fix: " . $e->getMessage());
            Log::error("Certificate fix error: " . $e->getMessage());
            Log::error($e->getTraceAsString());
            
            return 1;
        }
    }

    /**
     * Get certificates from SQL file for a specific vessel
     */
    private function getCertificatesFromSql($vesselId)
    {
        $sqlFilePath = base_path('aikhrrte_certificates.sql');
        
        if (!file_exists($sqlFilePath)) {
            $this->error("SQL file not found at: {$sqlFilePath}");
            return [];
        }

        $sqlContent = file_get_contents($sqlFilePath);
        
        // Extract vessel_certificates data
        preg_match('/INSERT INTO `vessel_certificates` \(`id`, `vessel_id`, `certificate_id`, `id_no`, `vessel_department`, `issued_date`, `expiry_date`, `attachment_path`, `created_at`, `updated_at`\) VALUES(.*?);/s', $sqlContent, $matches);
        
        if (empty($matches)) {
            $this->error("No vessel_certificates data found in SQL file");
            return [];
        }

        $valuesString = $matches[1];
        
        // Parse values
        preg_match_all('/\((.*?)\)(?:,|$)/s', $valuesString, $valueMatches);
        $values = $valueMatches[1];
        
        $certificates = [];
        
        foreach ($values as $value) {
            $fields = str_getcsv($value, ',', "'");
            
            // Clean up fields
            foreach ($fields as &$field) {
                $field = trim($field);
                if ($field === 'NULL' || $field === 'null') {
                    $field = null;
                }
            }
            
            // Extract data
            $id = (int)$fields[0];
            $vId = (int)$fields[1];
            
            if ($vId === $vesselId) {
                $certificates[] = [
                    'id' => $id,
                    'vessel_id' => $vId,
                    'certificate_id' => (int)$fields[2],
                    'id_no' => $fields[3],
                    'vessel_department' => $fields[4],
                    'issued_date' => $fields[5] !== null ? $fields[5] : null,
                    'expiry_date' => $fields[6] !== null ? $fields[6] : null,
                    'attachment_path' => $fields[7],
                ];
            }
        }
        
        return $certificates;
    }

    /**
     * Import certificates from SQL file for a specific vessel
     */
    private function importCertificatesFromSql($vesselId, $certificates)
    {
        $importedCount = 0;
        
        foreach ($certificates as $certificate) {
            // Check if certificate already exists
            $exists = VesselCertificate::where('vessel_id', $vesselId)
                ->where('certificate_id', $certificate['certificate_id'])
                ->exists();
                
            if (!$exists) {
                // Create new vessel certificate
                VesselCertificate::create([
                    'vessel_id' => $vesselId,
                    'certificate_id' => $certificate['certificate_id'],
                    'id_no' => $certificate['id_no'],
                    'vessel_department' => $certificate['vessel_department'],
                    'issued_date' => $certificate['issued_date'],
                    'expiry_date' => $certificate['expiry_date'],
                    'attachment_path' => $certificate['attachment_path'],
                ]);
                
                $importedCount++;
            }
        }
        
        return $importedCount;
    }
}
