<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Vessel;
use App\Models\Certificate;
use App\Models\VesselCertificate;

class FixZeroCertificateVessels extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'vessels:fix-zero-certificates';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fix vessels with 0 certificates by adding default certificates';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting to fix vessels with 0 certificates...');

        // Get all vessels
        $vessels = Vessel::all();
        $this->info("Found {$vessels->count()} vessels");

        // Get all certificate types
        $certificates = Certificate::all();
        $this->info("Found {$certificates->count()} certificate types");

        // Check if required certificates exist for default data
        $corCertificate = Certificate::find(21); // COR certificate
        $dmlcCertificate = Certificate::find(26); // DMLC certificate

        if (!$corCertificate) {
            $this->warn("COR certificate (ID: 21) not found in database - will not set default data");
        }

        if (!$dmlcCertificate) {
            $this->warn("DMLC certificate (ID: 26) not found in database - will not set default data");
        }

        // Begin transaction
        DB::beginTransaction();

        try {
            $fixedCount = 0;
            $alreadyOkCount = 0;
            $createdCertificatesCount = 0;

            foreach ($vessels as $vessel) {
                $certificateCount = VesselCertificate::where('vessel_id', $vessel->id)->count();

                if ($certificateCount === 0) {
                    // This vessel has 0 certificates, let's add all certificate types
                    $this->info("Fixing vessel with 0 certificates: {$vessel->name} (ID: {$vessel->id})");

                    // Create empty certificates for all certificate types
                    foreach ($certificates as $certificate) {
                        VesselCertificate::create([
                            'vessel_id' => $vessel->id,
                            'certificate_id' => $certificate->id,
                        ]);
                        $createdCertificatesCount++;
                    }

                    // Add default data to COR and DMLC certificates if they exist
                    if ($corCertificate) {
                        $corVesselCertificate = VesselCertificate::where('vessel_id', $vessel->id)
                            ->where('certificate_id', $corCertificate->id)
                            ->first();

                        if ($corVesselCertificate) {
                            $corVesselCertificate->update([
                                'id_no' => 'DEFAULT-COR-' . $vessel->id,
                                'vessel_department' => 'FLAG',
                                'issued_date' => now(),
                                'expiry_date' => now()->addYears(5),
                            ]);
                        }
                    }

                    if ($dmlcCertificate) {
                        $dmlcVesselCertificate = VesselCertificate::where('vessel_id', $vessel->id)
                            ->where('certificate_id', $dmlcCertificate->id)
                            ->first();

                        if ($dmlcVesselCertificate) {
                            $dmlcVesselCertificate->update([
                                'id_no' => 'DEFAULT-DMLC-' . $vessel->id,
                                'vessel_department' => 'FLAG',
                                'issued_date' => now(),
                                'expiry_date' => now()->addYears(5),
                            ]);
                        }
                    }

                    $fixedCount++;
                } else {
                    $this->line("Vessel {$vessel->name} (ID: {$vessel->id}) already has {$certificateCount} certificates");
                    $alreadyOkCount++;
                }
            }

            // Commit transaction
            DB::commit();

            $this->info("Fix completed successfully!");
            $this->info("Fixed {$fixedCount} vessels with 0 certificates");
            $this->info("Created {$createdCertificatesCount} certificates");
            $this->info("{$alreadyOkCount} vessels already had certificates");

            return 0;
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();

            $this->error("Error during fix: " . $e->getMessage());
            Log::error("Certificate fix error: " . $e->getMessage());
            Log::error($e->getTraceAsString());

            return 1;
        }
    }
}
