<?php

namespace App\Http\Controllers;

use App\Models\Vessel;
use App\Models\VesselPlan;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class VesselPlanController extends Controller
{
    /**
     * Store a newly created vessel plan in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'plan_id' => 'required|exists:plans,id',
            'attachment' => 'nullable|file|max:32768', // 32MB
        ]);

        // Check if vessel already has this plan
        $existingPlan = VesselPlan::where('vessel_id', $request->vessel_id)
            ->where('plan_id', $request->plan_id)
            ->first();

        if ($existingPlan) {
            return redirect()->back()
                ->with('error', 'This vessel already has this plan. Please edit the existing plan instead.');
        }

        $vesselPlan = new VesselPlan();
        $vesselPlan->vessel_id = $request->vessel_id;
        $vesselPlan->plan_id = $request->plan_id;

        // Handle file upload
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'plans');
            $path = $file->storeAs('plans', $fileName, 'public');
            $vesselPlan->attachment_path = $path;
        }

        $vesselPlan->save();

        return redirect()->route('vessels.show', $request->vessel_id)
            ->with('success', 'Vessel plan added successfully.');
    }

    /**
     * Update the specified vessel plan in storage.
     */
    public function update(Request $request, VesselPlan $vesselPlan)
    {
        $request->validate([
            'plan_id' => 'required|exists:plans,id',
            'attachment' => 'nullable|file|max:32768', // 32MB
        ]);

        $vesselPlan->plan_id = $request->plan_id;

        // Handle file upload
        if ($request->hasFile('attachment')) {
            // Delete old file if exists
            if ($vesselPlan->attachment_path) {
                Storage::disk('public')->delete($vesselPlan->attachment_path);
            }

            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'plans');
            $path = $file->storeAs('plans', $fileName, 'public');
            $vesselPlan->attachment_path = $path;
        }

        $vesselPlan->save();

        return redirect()->route('vessels.show', $vesselPlan->vessel_id)
            ->with('success', 'Vessel plan updated successfully.');
    }

    /**
     * Remove the specified vessel plan from storage.
     */
    public function destroy(VesselPlan $vesselPlan)
    {
        // Delete attachment if exists
        if ($vesselPlan->attachment_path) {
            Storage::disk('public')->delete($vesselPlan->attachment_path);
        }

        $vesselId = $vesselPlan->vessel_id;
        $vesselPlan->delete();

        return redirect()->route('vessels.show', $vesselId)
            ->with('success', 'Vessel plan deleted successfully.');
    }

    /**
     * Generate a unique file name by checking for existing files and adding _1, _2, etc.
     */
    private function generateUniqueFileName($originalName, $directory)
    {
        $fileName = $originalName;
        $counter = 1;

        // Keep checking until we find a unique name
        while (\Illuminate\Support\Facades\Storage::disk('public')->exists($directory . '/' . $fileName)) {
            $pathInfo = pathinfo($originalName);
            $name = $pathInfo['filename'];
            $extension = isset($pathInfo['extension']) ? '.' . $pathInfo['extension'] : '';

            $fileName = $name . '_' . $counter . $extension;
            $counter++;
        }

        return $fileName;
    }
}
