<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class VesselCertificate extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'vessel_id',
        'certificate_id',
        'id_no',
        'vessel_department',
        'issued_date',
        'expiry_date',
        'attachment_path',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'issued_date' => 'date',
        'expiry_date' => 'date',
        'attachment_path' => 'json',
    ];

    /**
     * Get the vessel that owns the certificate.
     */
    public function vessel()
    {
        return $this->belongsTo(Vessel::class);
    }

    /**
     * Get the certificate that owns the vessel certificate.
     */
    public function certificate()
    {
        return $this->belongsTo(Certificate::class);
    }

    /**
     * Check if the certificate is expired.
     *
     * @return bool
     */
    public function isExpired()
    {
        if (!$this->expiry_date) {
            return false;
        }

        return $this->expiry_date->isPast();
    }

    /**
     * Check if the certificate is expiring soon based on warning days.
     *
     * @param int $warningDays
     * @return bool
     */
    public function isExpiringSoon($warningDays = 30)
    {
        if (!$this->expiry_date) {
            return false;
        }

        $warningDate = now()->addDays($warningDays);
        return $this->expiry_date->lte($warningDate) && $this->expiry_date->gte(now());
    }
}
