<?php
// This script imports vessel particulars and surveys from the aikhrrte_certificates.sql file

// Load the Laravel environment
require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Models\Vessel;
use App\Models\Survey;
use App\Models\VesselSurvey;
use App\Models\VesselParticular;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

// Function to parse SQL file and extract vessel particulars
function extractVesselParticularsFromSql($sqlFilePath) {
    if (!file_exists($sqlFilePath)) {
        echo "SQL file not found at: $sqlFilePath\n";
        return [];
    }

    $sqlContent = file_get_contents($sqlFilePath);

    // Extract the INSERT statements for vessel_particulars
    preg_match_all('/INSERT INTO `vessel_particulars`.*VALUES\s*(.*?);/s', $sqlContent, $allMatches);

    $particulars = [];

    foreach ($allMatches[1] as $valuesString) {
        // Parse values
        preg_match_all('/\((\d+),\s*(\d+),\s*([^,]*),\s*(.*?)\)(?:,|$)/s', $valuesString, $valueMatches, PREG_SET_ORDER);

        foreach ($valueMatches as $match) {
            // Skip if not enough fields
            if (count($match) < 5) {
                echo "Skipping row with insufficient fields: " . count($match) . "\n";
                continue;
            }

            // Clean up fields
            $id = (int)$match[1];
            $vesselId = (int)$match[2];

            $imoNumber = trim($match[3], "'");
            if ($imoNumber === 'NULL' || $imoNumber === 'null') $imoNumber = null;

            $attachmentPath = trim($match[4], "'");
            if ($attachmentPath === 'NULL' || $attachmentPath === 'null') {
                $attachmentPath = null;
            }

            $particulars[] = [
                'id' => $id,
                'vessel_id' => $vesselId,
                'imo_number' => $imoNumber,
                'attachment_path' => $attachmentPath,
            ];
        }
    }

    if (empty($particulars)) {
        echo "No vessel_particulars data found in SQL file\n";
    }

    return $particulars;
}

// Function to parse SQL file and extract vessel surveys
function extractVesselSurveysFromSql($sqlFilePath) {
    if (!file_exists($sqlFilePath)) {
        echo "SQL file not found at: $sqlFilePath\n";
        return [];
    }

    $sqlContent = file_get_contents($sqlFilePath);

    // Extract the INSERT statements for vessel_surveys
    preg_match_all('/INSERT INTO `vessel_surveys`.*VALUES\s*(.*?);/s', $sqlContent, $allMatches);

    $surveys = [];

    foreach ($allMatches[1] as $valuesString) {
        // Parse values
        preg_match_all('/\((\d+),\s*(\d+),\s*(\d+),\s*(.*?),\s*(.*?),\s*(.*?),\s*(.*?),\s*(.*?)\)(?:,|$)/s', $valuesString, $valueMatches, PREG_SET_ORDER);

        foreach ($valueMatches as $match) {
            // Skip if not enough fields
            if (count($match) < 9) {
                echo "Skipping row with insufficient fields: " . count($match) . "\n";
                continue;
            }

            // Clean up fields
            $id = (int)$match[1];
            $vesselId = (int)$match[2];
            $surveyId = (int)$match[3];

            $lastDate = trim($match[4], "'");
            if ($lastDate === 'NULL' || $lastDate === 'null' || $lastDate === '0000-00-00') $lastDate = null;

            $dueDate = trim($match[5], "'");
            if ($dueDate === 'NULL' || $dueDate === 'null' || $dueDate === '0000-00-00') $dueDate = null;

            $windowDate = trim($match[6], "'");
            if ($windowDate === 'NULL' || $windowDate === 'null' || $windowDate === '0000-00-00') $windowDate = null;

            $status = trim($match[7], "'");
            if ($status === 'NULL' || $status === 'null') $status = null;

            $generalNote = trim($match[8], "'");
            if ($generalNote === 'NULL' || $generalNote === 'null') $generalNote = null;

            $surveys[] = [
                'id' => $id,
                'vessel_id' => $vesselId,
                'survey_id' => $surveyId,
                'last_date' => $lastDate,
                'due_date' => $dueDate,
                'window_date' => $windowDate,
                'status' => $status,
                'general_note' => $generalNote,
            ];
        }
    }

    if (empty($surveys)) {
        echo "No vessel_surveys data found in SQL file\n";
    }

    return $surveys;
}

// Main execution
echo "Starting import of vessel particulars and surveys from aikhrrte_certificates.sql...\n";

// Get all vessels
$vessels = Vessel::all();
echo "Found " . $vessels->count() . " vessels\n";

// Get all surveys
$surveys = Survey::all();
echo "Found " . $surveys->count() . " survey types\n";

// Extract particulars and surveys from SQL file
$sqlFilePath = __DIR__ . '/aikhrrte_certificates.sql';
$sqlParticulars = extractVesselParticularsFromSql($sqlFilePath);
echo "Extracted " . count($sqlParticulars) . " vessel particulars from SQL file\n";

$sqlSurveys = extractVesselSurveysFromSql($sqlFilePath);
echo "Extracted " . count($sqlSurveys) . " vessel surveys from SQL file\n";

// Begin transaction
DB::beginTransaction();

try {
    // Import vessel particulars
    $particularImportCount = 0;
    $particularUpdateCount = 0;
    $particularSkipCount = 0;

    foreach ($sqlParticulars as $particular) {
        $vesselId = $particular['vessel_id'];

        // Check if vessel exists
        $vessel = Vessel::find($vesselId);
        if (!$vessel) {
            echo "  Vessel ID {$vesselId} not found, skipping particular\n";
            $particularSkipCount++;
            continue;
        }

        // Check if particular already exists
        $existingParticular = VesselParticular::where('vessel_id', $vesselId)->first();

        if ($existingParticular) {
            // Update existing particular
            $existingParticular->imo_number = $particular['imo_number'];

            // Only update attachment_path if it's not null
            if ($particular['attachment_path']) {
                $existingParticular->attachment_path = $particular['attachment_path'];
            }

            $existingParticular->save();

            $particularUpdateCount++;
            echo "  Updated particular for vessel ID {$vesselId}\n";
        } else {
            // Create new particular
            VesselParticular::create([
                'vessel_id' => $vesselId,
                'imo_number' => $particular['imo_number'],
                'attachment_path' => $particular['attachment_path'],
            ]);

            $particularImportCount++;
            echo "  Created particular for vessel ID {$vesselId}\n";
        }
    }

    // Import vessel surveys
    $surveyImportCount = 0;
    $surveyUpdateCount = 0;
    $surveySkipCount = 0;

    foreach ($sqlSurveys as $survey) {
        $vesselId = $survey['vessel_id'];
        $surveyId = $survey['survey_id'];

        // Check if vessel exists
        $vessel = Vessel::find($vesselId);
        if (!$vessel) {
            echo "  Vessel ID {$vesselId} not found, skipping survey\n";
            $surveySkipCount++;
            continue;
        }

        // Check if survey type exists
        $surveyType = Survey::find($surveyId);
        if (!$surveyType) {
            echo "  Survey ID {$surveyId} not found, skipping survey\n";
            $surveySkipCount++;
            continue;
        }

        // Check if survey already exists
        $existingSurvey = VesselSurvey::where('vessel_id', $vesselId)
            ->where('survey_id', $surveyId)
            ->first();

        if ($existingSurvey) {
            // Update existing survey
            $existingSurvey->last_date = $survey['last_date'];
            $existingSurvey->due_date = $survey['due_date'];
            $existingSurvey->window_date = $survey['window_date'];
            $existingSurvey->status = $survey['status'];
            $existingSurvey->general_note = $survey['general_note'];
            $existingSurvey->save();

            $surveyUpdateCount++;
            echo "  Updated survey for vessel ID {$vesselId}, survey ID {$surveyId}\n";
        } else {
            // Create new survey
            VesselSurvey::create([
                'vessel_id' => $vesselId,
                'survey_id' => $surveyId,
                'last_date' => $survey['last_date'],
                'due_date' => $survey['due_date'],
                'window_date' => $survey['window_date'],
                'status' => $survey['status'],
                'general_note' => $survey['general_note'],
            ]);

            $surveyImportCount++;
            echo "  Created survey for vessel ID {$vesselId}, survey ID {$surveyId}\n";
        }
    }

    // Commit transaction
    DB::commit();

    echo "Import completed successfully!\n";
    echo "Particulars: Created: {$particularImportCount}, Updated: {$particularUpdateCount}, Skipped: {$particularSkipCount}\n";
    echo "Surveys: Created: {$surveyImportCount}, Updated: {$surveyUpdateCount}, Skipped: {$surveySkipCount}\n";

} catch (\Exception $e) {
    // Rollback transaction on error
    DB::rollBack();

    echo "Error during import: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}
