<?php
/**
 * Direct script to delete certificate attachments
 * This is a standalone script that doesn't rely on Laravel's authentication
 */

// Set error reporting to log errors but not display them
error_reporting(E_ALL);
ini_set('display_errors', 0);

// Always set the content type to JSON
header('Content-Type: application/json');

// Function to return a JSON error response
function returnJsonError($message, $code = 500) {
    echo json_encode([
        'success' => false,
        'message' => $message,
        'logs' => isset($GLOBALS['logs']) ? $GLOBALS['logs'] : []
    ]);
    exit;
}

// Initialize logs array
$logs = [];

try {
    // Database connection parameters
    $host = 'localhost';
    $dbname = 'vessel_certificates';
    $username = 'root';
    $password = '';
    
    // Try to get database credentials from Laravel's .env file
    if (file_exists(__DIR__ . '/../.env')) {
        $envFile = file_get_contents(__DIR__ . '/../.env');
        preg_match('/DB_HOST=([^\n]+)/', $envFile, $hostMatches);
        preg_match('/DB_DATABASE=([^\n]+)/', $envFile, $dbnameMatches);
        preg_match('/DB_USERNAME=([^\n]+)/', $envFile, $usernameMatches);
        preg_match('/DB_PASSWORD=([^\n]+)/', $envFile, $passwordMatches);
        
        if (!empty($hostMatches[1])) $host = trim($hostMatches[1]);
        if (!empty($dbnameMatches[1])) $dbname = trim($dbnameMatches[1]);
        if (!empty($usernameMatches[1])) $username = trim($usernameMatches[1]);
        if (!empty($passwordMatches[1])) $password = trim($passwordMatches[1]);
    }
    
    $logs[] = "Connecting to database: $host, $dbname";
    
    // Connect to the database
    $pdo = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $logs[] = "Connected to database successfully";
    
    // Get the request parameters
    $vesselCertificateId = $_POST['vessel_certificate_id'] ?? null;
    $attachmentIndex = isset($_POST['attachment_index']) ? intval($_POST['attachment_index']) : null;
    
    if (!$vesselCertificateId) {
        returnJsonError('Missing vessel_certificate_id parameter');
    }
    
    if ($attachmentIndex === null) {
        returnJsonError('Missing attachment_index parameter');
    }
    
    $logs[] = "Processing deletion for certificate ID: $vesselCertificateId, attachment index: $attachmentIndex";
    
    // Get the vessel certificate from the database
    $stmt = $pdo->prepare("SELECT * FROM vessel_certificates WHERE id = ?");
    $stmt->execute([$vesselCertificateId]);
    $vesselCertificate = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$vesselCertificate) {
        returnJsonError('Certificate not found');
    }
    
    $logs[] = "Found certificate: " . json_encode($vesselCertificate);
    
    // Function to aggressively delete a file
    function deleteFileAggressively($path, &$logs) {
        $logs[] = "Attempting to delete: $path";
        
        // Try all possible storage locations
        $possiblePaths = [
            // Storage paths
            __DIR__ . '/../storage/app/public/' . $path,
            __DIR__ . '/../storage/app/public/certificates/' . basename($path),
            __DIR__ . '/../storage/app/public/' . ltrim($path, '/'),
            
            // Public paths
            __DIR__ . '/storage/' . $path,
            __DIR__ . '/storage/certificates/' . basename($path),
            __DIR__ . '/storage/' . ltrim($path, '/'),
            __DIR__ . '/' . $path,
            
            // Absolute paths
            $path,
            '/var/www/html/storage/app/public/' . $path,
            '/var/www/html/public/storage/' . $path,
        ];
        
        $deleted = false;
        foreach ($possiblePaths as $fullPath) {
            if (file_exists($fullPath)) {
                $logs[] = "Found file at: $fullPath";
                if (unlink($fullPath)) {
                    $logs[] = "Successfully deleted: $fullPath";
                    $deleted = true;
                } else {
                    $logs[] = "Failed to delete: $fullPath";
                }
            }
        }
        
        return $deleted;
    }
    
    // Process the deletion
    if ($vesselCertificate['attachment_path']) {
        // Handle both string and array formats
        $attachmentPath = $vesselCertificate['attachment_path'];
        $logs[] = "Original attachment path: " . json_encode($attachmentPath);
        
        if (is_string($attachmentPath) && !is_array(json_decode($attachmentPath, true))) {
            // Single attachment as string
            if ($attachmentIndex == 0) {
                $attachmentToDelete = $attachmentPath;
                $logs[] = "Deleting single attachment: $attachmentToDelete";
                
                // Delete the file
                deleteFileAggressively($attachmentToDelete, $logs);
                
                // Update database
                $stmt = $pdo->prepare("UPDATE vessel_certificates SET attachment_path = NULL WHERE id = ?");
                $stmt->execute([$vesselCertificateId]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Attachment deleted successfully.',
                    'logs' => $logs
                ]);
                exit;
            } else {
                returnJsonError('Invalid attachment index for single attachment');
            }
        } else {
            // Multiple attachments as array
            $attachments = is_array($attachmentPath) ? $attachmentPath : json_decode($attachmentPath, true);
            
            if (is_array($attachments) && isset($attachments[$attachmentIndex])) {
                $attachmentToDelete = $attachments[$attachmentIndex];
                $logs[] = "Deleting attachment from array: $attachmentToDelete";
                
                // Delete the file
                deleteFileAggressively($attachmentToDelete, $logs);
                
                // Remove from array
                unset($attachments[$attachmentIndex]);
                
                // Reindex array
                $attachments = array_values($attachments);
                
                // Update database
                $newAttachmentPath = !empty($attachments) ? json_encode($attachments) : null;
                $stmt = $pdo->prepare("UPDATE vessel_certificates SET attachment_path = ? WHERE id = ?");
                $stmt->execute([$newAttachmentPath, $vesselCertificateId]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Attachment deleted successfully.',
                    'logs' => $logs
                ]);
                exit;
            } else {
                returnJsonError('Attachment not found in array');
            }
        }
    } else {
        returnJsonError('No attachments found');
    }
} catch (Exception $e) {
    // Log the error
    error_log('Error in direct_delete_attachment.php: ' . $e->getMessage());
    
    // Return a JSON error response
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while deleting the attachment: ' . $e->getMessage(),
        'logs' => isset($logs) ? $logs : []
    ]);
    exit;
}
