<?php

namespace App\Http\Controllers;

use App\Models\Certificate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CertificateController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Certificate::query();

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'LIKE', "%{$search}%")
                  ->orWhere('category', 'LIKE', "%{$search}%");
        }

        // Filter by category
        if ($request->has('category') && $request->category !== 'all') {
            $query->where('category', $request->category);
        }

        $certificates = $query->orderBy('category')
                             ->orderBy('display_order')
                             ->get();

        // Get all categories for filter dropdown
        $categories = Certificate::select('category')
                                ->distinct()
                                ->orderBy('category')
                                ->pluck('category');

        return view('certificates.index', compact('certificates', 'categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('certificates.index')
                ->with('error', 'You do not have permission to create certificates.');
        }

        // Get all categories for dropdown
        $categories = Certificate::select('category')
                                ->distinct()
                                ->orderBy('category')
                                ->pluck('category');

        return view('certificates.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('certificates.index')
                ->with('error', 'You do not have permission to create certificates.');
        }

        $request->validate([
            'name' => 'required|string|max:100|unique:certificates',
            'category' => 'required|string|max:100',
            'display_order' => 'required|integer|min:0',
        ]);

        $certificate = Certificate::create([
            'name' => $request->name,
            'category' => $request->category,
            'display_order' => $request->display_order,
            'category_order' => $request->category_order ?? 1000,
        ]);

        return redirect()->route('certificates.index')
            ->with('success', 'Certificate created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Certificate $certificate)
    {
        // Get all vessels that have this certificate
        $vesselCertificates = $certificate->vesselCertificates()
                                        ->with('vessel')
                                        ->get();

        return view('certificates.show', compact('certificate', 'vesselCertificates'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Certificate $certificate)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('certificates.index')
                ->with('error', 'You do not have permission to edit certificates.');
        }

        // Get all categories for dropdown
        $categories = Certificate::select('category')
                                ->distinct()
                                ->orderBy('category')
                                ->pluck('category');

        return view('certificates.edit', compact('certificate', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Certificate $certificate)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('certificates.index')
                ->with('error', 'You do not have permission to edit certificates.');
        }

        $request->validate([
            'name' => 'required|string|max:100|unique:certificates,name,' . $certificate->id,
            'category' => 'required|string|max:100',
            'display_order' => 'required|integer|min:0',
        ]);

        $certificate->update([
            'name' => $request->name,
            'category' => $request->category,
            'display_order' => $request->display_order,
            'category_order' => $request->category_order ?? $certificate->category_order,
        ]);

        return redirect()->route('certificates.index')
            ->with('success', 'Certificate updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Certificate $certificate)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('certificates.index')
                ->with('error', 'You do not have permission to delete certificates.');
        }

        // Check if certificate is in use
        $inUseCount = $certificate->vesselCertificates()->count();
        if ($inUseCount > 0) {
            return redirect()->route('certificates.index')
                ->with('error', "Cannot delete certificate. It is currently in use by {$inUseCount} vessels.");
        }

        $certificate->delete();

        return redirect()->route('certificates.index')
            ->with('success', 'Certificate deleted successfully.');
    }

    /**
     * Update certificate date.
     */
    public function updateDate(Request $request)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to update certificates.');
        }

        $request->validate([
            'vessel_certificate_id' => 'required|exists:vessel_certificates,id',
            'date_type' => 'required|in:issued_date,expiry_date',
            'date_value' => 'nullable|date',
        ]);

        $vesselCertificate = \App\Models\VesselCertificate::findOrFail($request->vessel_certificate_id);

        if ($request->date_type === 'issued_date') {
            $vesselCertificate->issued_date = $request->date_value;
        } else {
            $vesselCertificate->expiry_date = $request->date_value;
        }

        $vesselCertificate->save();

        return redirect()->back()
            ->with('success', 'Certificate date updated successfully.');
    }

    /**
     * Update certificate details.
     */
    public function updateCertificate(Request $request)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to update certificates.');
        }

        $request->validate([
            'vessel_certificate_id' => 'required|exists:vessel_certificates,id',
            'id_no' => 'nullable|string|max:100',
            'vessel_department' => 'nullable|string|max:100',
        ]);

        $vesselCertificate = \App\Models\VesselCertificate::findOrFail($request->vessel_certificate_id);

        $vesselCertificate->id_no = $request->id_no;
        $vesselCertificate->vessel_department = $request->vessel_department;
        $vesselCertificate->save();

        return redirect()->back()
            ->with('success', 'Certificate details updated successfully.');
    }

    /**
     * Simple method to remove an attachment - this is a more direct approach
     * that should work even when the session expires
     */
    public function removeAttachment(Request $request)
    {
        try {
            // Log the request for debugging
            \Illuminate\Support\Facades\Log::info('Remove attachment request:', $request->all());

            $request->validate([
                'vessel_certificate_id' => 'required|exists:vessel_certificates,id',
                'attachment_index' => 'required|integer|min:0',
                'attachment_path' => 'required|string',
            ]);

            $vesselCertificate = \App\Models\VesselCertificate::findOrFail($request->vessel_certificate_id);

            // Try to delete the file
            $this->deleteFile($request->attachment_path);

            // Update the database
            if ($vesselCertificate->attachment_path) {
                if (is_string($vesselCertificate->attachment_path) && !is_array(json_decode($vesselCertificate->attachment_path, true))) {
                    // Single attachment as string
                    if ($request->attachment_index == 0) {
                        $vesselCertificate->attachment_path = null;
                        $vesselCertificate->save();
                    }
                } else {
                    // Multiple attachments as array
                    $attachments = is_array($vesselCertificate->attachment_path)
                        ? $vesselCertificate->attachment_path
                        : json_decode($vesselCertificate->attachment_path, true);

                    if (is_array($attachments) && isset($attachments[$request->attachment_index])) {
                        // Remove from array
                        unset($attachments[$request->attachment_index]);

                        // Reindex array
                        $attachments = array_values($attachments);

                        // Update database
                        $vesselCertificate->attachment_path = !empty($attachments) ? json_encode($attachments) : null;
                        $vesselCertificate->save();
                    }
                }
            }

            // Check if it's an AJAX request
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Attachment deleted successfully.'
                ]);
            }

            // Redirect back with success message
            return redirect()->back()->with('success', 'Attachment deleted successfully.');
        } catch (\Exception $e) {
            // Log the error
            \Illuminate\Support\Facades\Log::error('Error removing attachment: ' . $e->getMessage());

            // Check if it's an AJAX request
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error removing attachment: ' . $e->getMessage()
                ], 500);
            }

            // Redirect back with error message
            return redirect()->back()->with('error', 'Error removing attachment: ' . $e->getMessage());
        }
    }

    /**
     * Bulk download certificates as ZIP file
     */
    public function bulkDownload(Request $request)
    {
        try {
            $request->validate([
                'certificate_ids' => 'required|array',
                'certificate_ids.*' => 'exists:vessel_certificates,id',
                'download_type' => 'required|in:selected,category',
                'category' => 'nullable|string',
                'vessel_name' => 'required|string'
            ]);

            $vesselCertificates = \App\Models\VesselCertificate::whereIn('id', $request->certificate_ids)
                ->whereNotNull('attachment_path')
                ->with(['vessel', 'certificate'])
                ->get();

            if ($vesselCertificates->isEmpty()) {
                return redirect()->back()->with('error', 'No certificates with attachments found.');
            }

            // Create a temporary directory for the ZIP file
            $tempDir = storage_path('app/temp');
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }

            $zipFileName = $request->vessel_name . '_certificates_' . date('Y-m-d_H-i-s') . '.zip';
            if ($request->download_type === 'category' && $request->category) {
                $zipFileName = $request->vessel_name . '_' . str_replace(' ', '_', $request->category) . '_certificates_' . date('Y-m-d_H-i-s') . '.zip';
            }

            $zipPath = $tempDir . '/' . $zipFileName;

            $zip = new \ZipArchive();
            if ($zip->open($zipPath, \ZipArchive::CREATE) !== TRUE) {
                return redirect()->back()->with('error', 'Could not create ZIP file.');
            }

            $addedFiles = 0;

            foreach ($vesselCertificates as $vesselCert) {
                $attachments = $this->getAttachmentPaths($vesselCert->attachment_path);

                foreach ($attachments as $index => $attachmentPath) {
                    $fullPath = $this->getFullAttachmentPath($attachmentPath);

                    if (file_exists($fullPath)) {
                        // Create a meaningful filename
                        $extension = pathinfo($attachmentPath, PATHINFO_EXTENSION);
                        $fileName = $vesselCert->certificate->name;

                        // Add index if multiple attachments
                        if (count($attachments) > 1) {
                            $fileName .= '_' . ($index + 1);
                        }

                        $fileName .= '.' . $extension;

                        // Sanitize filename
                        $fileName = preg_replace('/[^a-zA-Z0-9._-]/', '_', $fileName);

                        // Add to ZIP
                        $zip->addFile($fullPath, $fileName);
                        $addedFiles++;
                    }
                }
            }

            $zip->close();

            if ($addedFiles === 0) {
                unlink($zipPath);
                return redirect()->back()->with('error', 'No attachment files found on the server.');
            }

            // Return the ZIP file for download
            return response()->download($zipPath, $zipFileName)->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Error in bulk download: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error creating download: ' . $e->getMessage());
        }
    }

    /**
     * Get attachment paths from the stored data
     */
    private function getAttachmentPaths($attachmentPath)
    {
        if (!$attachmentPath) {
            return [];
        }

        // Try to decode as JSON first
        $decoded = json_decode($attachmentPath, true);
        if (is_array($decoded)) {
            return $decoded;
        }

        // If not JSON, treat as single path
        return [$attachmentPath];
    }

    /**
     * Get the full path to an attachment file
     */
    private function getFullAttachmentPath($attachmentPath)
    {
        // Try different possible locations
        $possiblePaths = [
            storage_path('app/public/' . $attachmentPath),
            storage_path('app/public/certificates/' . basename($attachmentPath)),
            public_path('storage/' . $attachmentPath),
            public_path('storage/certificates/' . basename($attachmentPath)),
        ];

        foreach ($possiblePaths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }

        return $attachmentPath; // Return original if not found
    }

    /**
     * Delete certificate attachment.
     */
    public function deleteAttachment(Request $request)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to delete attachments.');
        }

        $request->validate([
            'vessel_certificate_id' => 'required|exists:vessel_certificates,id',
            'attachment_index' => 'required|integer|min:0',
        ]);

        $vesselCertificate = \App\Models\VesselCertificate::findOrFail($request->vessel_certificate_id);

        if ($vesselCertificate->attachment_path) {
            // Handle both string and array formats
            if (is_string($vesselCertificate->attachment_path) && !is_array(json_decode($vesselCertificate->attachment_path, true))) {
                // Single attachment as string
                if ($request->attachment_index == 0) {
                    $attachmentToDelete = $vesselCertificate->attachment_path;

                    // Delete the file - handle different path formats
                    $this->deleteFile($attachmentToDelete);

                    // Update database
                    $vesselCertificate->attachment_path = null;
                    $vesselCertificate->save();

                    // Check if it's an AJAX request
                    if ($request->ajax() || $request->wantsJson()) {
                        return response()->json([
                            'success' => true,
                            'message' => 'Attachment deleted successfully.'
                        ]);
                    }

                    return redirect()->back()
                        ->with('success', 'Attachment deleted successfully.');
                }
            } else {
                // Multiple attachments as array
                $attachments = is_array($vesselCertificate->attachment_path)
                    ? $vesselCertificate->attachment_path
                    : json_decode($vesselCertificate->attachment_path, true);

                if (is_array($attachments) && isset($attachments[$request->attachment_index])) {
                    $attachmentToDelete = $attachments[$request->attachment_index];

                    // Delete the file - handle different path formats
                    $this->deleteFile($attachmentToDelete);

                    // Remove from array
                    unset($attachments[$request->attachment_index]);

                    // Reindex array
                    $attachments = array_values($attachments);

                    // Update database
                    $vesselCertificate->attachment_path = !empty($attachments) ? json_encode($attachments) : null;
                    $vesselCertificate->save();

                    // Check if it's an AJAX request
                    if ($request->ajax() || $request->wantsJson()) {
                        return response()->json([
                            'success' => true,
                            'message' => 'Attachment deleted successfully.'
                        ]);
                    }

                    return redirect()->back()
                        ->with('success', 'Attachment deleted successfully.');
                }
            }
        }

        // Check if it's an AJAX request
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => false,
                'message' => 'Attachment not found.'
            ], 404);
        }

        return redirect()->back()
            ->with('error', 'Attachment not found.');
    }

    /**
     * Helper method to delete a file with different path formats
     */
    public function deleteFile($path)
    {
        // Log the deletion attempt for debugging
        \Illuminate\Support\Facades\Log::info('Attempting to delete file: ' . $path);

        try {
            // Handle URLs with https://
            if (strpos($path, 'https://') === 0) {
                // Extract the path part from the URL
                $urlParts = parse_url($path);
                if (isset($urlParts['path'])) {
                    $pathOnly = ltrim($urlParts['path'], '/');
                    // Try to delete from public storage
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($pathOnly);

                    // Also try to delete from certificates folder
                    if (strpos($pathOnly, 'certificates/') !== false) {
                        \Illuminate\Support\Facades\Storage::disk('public')->delete($pathOnly);
                    }

                    // Try with certificates/uploads prefix
                    $altPath = 'certificates/uploads/' . basename($pathOnly);
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($altPath);
                }
            }

            // Handle paths starting with /certificates/uploads/
            if (strpos($path, '/certificates/uploads/') === 0) {
                // Try the cleaned path
                $cleanPath = str_replace('/certificates/uploads/', 'certificates/', $path);
                \Illuminate\Support\Facades\Storage::disk('public')->delete($cleanPath);

                // Also try the original path
                \Illuminate\Support\Facades\Storage::disk('public')->delete($path);

                // Try without leading slash
                $noLeadingSlash = ltrim($path, '/');
                \Illuminate\Support\Facades\Storage::disk('public')->delete($noLeadingSlash);

                // Try just the filename
                $justFilename = basename($path);
                \Illuminate\Support\Facades\Storage::disk('public')->delete('certificates/' . $justFilename);
            }

            // Handle standard paths
            \Illuminate\Support\Facades\Storage::disk('public')->delete($path);

            // Try with certificates prefix
            if (strpos($path, 'certificates/') !== 0 && strpos($path, '/certificates/') !== 0) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete('certificates/' . basename($path));
            }

            // For absolute safety, try to delete the file directly if it exists
            $possiblePaths = [
                storage_path('app/public/' . $path),
                storage_path('app/public/certificates/' . basename($path)),
                storage_path('app/public/' . ltrim($path, '/')),
                public_path('storage/' . $path),
                public_path('storage/certificates/' . basename($path)),
                public_path('storage/' . ltrim($path, '/'))
            ];

            foreach ($possiblePaths as $fullPath) {
                if (file_exists($fullPath)) {
                    \Illuminate\Support\Facades\Log::info('Found file at: ' . $fullPath);
                    unlink($fullPath);
                }
            }

            // Log the deletion completion
            \Illuminate\Support\Facades\Log::info('File deletion completed for: ' . $path);

            return true;
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Error deleting file: ' . $e->getMessage());
            return false;
        }
    }
}
