<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = User::query();

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('username', 'LIKE', "%{$search}%")
                  ->orWhere('email', 'LIKE', "%{$search}%");
        }

        $users = $query->orderBy('username')->get();
        return view('users.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('users.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'username' => 'required|string|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'is_admin' => 'boolean',
        ]);

        // Set permissions based on user type
        $permissions = [];

        if ($request->user_type === 'admin') {
            $permissions = [
                'vessels' => true,
                'certificates' => true,
                'warnings' => true,
                'manage_certificates' => true,
                'particulars' => true,
                'surveys' => true,
                'manage_surveys' => true,
                'plans' => true,
                'manage_plans' => true
            ];
            $isAdmin = true;
        } elseif ($request->user_type === 'view_only') {
            $permissions = [
                'vessels' => true,
                'certificates' => false,
                'warnings' => true,
                'manage_certificates' => false,
                'particulars' => true,
                'surveys' => true,
                'manage_surveys' => false,
                'plans' => true,
                'manage_plans' => false
            ];
            $isAdmin = false;
        } else {
            // Custom permissions
            $permissions = [
                'vessels' => $request->has('permission_vessels'),
                'certificates' => $request->has('permission_certificates'),
                'warnings' => $request->has('permission_warnings'),
                'manage_certificates' => $request->has('permission_manage_certificates'),
                'particulars' => $request->has('permission_particulars'),
                'surveys' => $request->has('permission_surveys'),
                'manage_surveys' => $request->has('permission_manage_surveys'),
                'plans' => $request->has('permission_plans'),
                'manage_plans' => $request->has('permission_manage_plans')
            ];
            $isAdmin = $request->is_admin ?? false;
        }

        User::create([
            'username' => $request->username,
            'password' => Hash::make($request->password),
            'is_admin' => $isAdmin,
            'permissions' => $permissions,
        ]);

        return redirect()->route('users.index')
            ->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        // Only allow users to view their own profile unless they're an admin
        if (!auth()->user()->is_admin && auth()->id() !== $user->id) {
            abort(403);
        }

        return view('users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        return view('users.edit', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'username' => [
                'required',
                'string',
                'max:255',
                Rule::unique('users')->ignore($user->id),
            ],
            'is_admin' => 'boolean',
        ]);

        // Set permissions based on user type
        $permissions = [];

        if ($request->user_type === 'admin') {
            $permissions = [
                'vessels' => true,
                'certificates' => true,
                'warnings' => true,
                'manage_certificates' => true,
                'particulars' => true,
                'surveys' => true,
                'manage_surveys' => true,
                'plans' => true,
                'manage_plans' => true
            ];
            $isAdmin = true;
        } elseif ($request->user_type === 'view_only') {
            $permissions = [
                'vessels' => true,
                'certificates' => false,
                'warnings' => true,
                'manage_certificates' => false,
                'particulars' => true,
                'surveys' => true,
                'manage_surveys' => false,
                'plans' => true,
                'manage_plans' => false
            ];
            $isAdmin = false;
        } else {
            // Custom permissions
            $permissions = [
                'vessels' => $request->has('permission_vessels'),
                'certificates' => $request->has('permission_certificates'),
                'warnings' => $request->has('permission_warnings'),
                'manage_certificates' => $request->has('permission_manage_certificates'),
                'particulars' => $request->has('permission_particulars'),
                'surveys' => $request->has('permission_surveys'),
                'manage_surveys' => $request->has('permission_manage_surveys'),
                'plans' => $request->has('permission_plans'),
                'manage_plans' => $request->has('permission_manage_plans')
            ];
            $isAdmin = $request->is_admin ?? false;
        }

        $user->update([
            'username' => $request->username,
            'is_admin' => $isAdmin,
            'permissions' => $permissions,
        ]);

        // Update password if provided
        if ($request->filled('password')) {
            $request->validate([
                'password' => 'string|min:8|confirmed',
            ]);

            $user->update([
                'password' => Hash::make($request->password),
            ]);
        }

        return redirect()->route('users.index')
            ->with('success', 'User updated successfully.');
    }

    /**
     * Update the user's password.
     */
    public function updatePassword(Request $request, User $user)
    {
        // Only allow users to update their own password unless they're an admin
        if (!auth()->user()->is_admin && auth()->id() !== $user->id) {
            abort(403);
        }

        $request->validate([
            'current_password' => 'required|string',
            'password' => 'required|string|min:8|confirmed',
        ]);

        // Check current password
        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'The current password is incorrect.']);
        }

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('users.show', $user)
            ->with('success', 'Password updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Prevent deleting yourself
        if (auth()->id() === $user->id) {
            return redirect()->route('users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        return redirect()->route('users.index')
            ->with('success', 'User deleted successfully.');
    }
}
