<?php

namespace App\Http\Controllers;

use App\Models\Vessel;
use App\Models\Certificate;
use App\Models\Plan;
use App\Models\Survey;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class VesselController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Vessel::query();

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'LIKE', "%{$search}%");
        }

        // Get vessels with certificate counts
        $vessels = $query->orderBy('name')->get();

        // Check for vessels with 0 certificates and fix them
        $this->fixVesselsWithZeroCertificates($vessels);

        return view('vessels.index', compact('vessels'));
    }

    /**
     * Fix vessels with 0 certificates by adding default certificates
     */
    private function fixVesselsWithZeroCertificates($vessels)
    {
        foreach ($vessels as $vessel) {
            $certificateCount = $vessel->vesselCertificates()->count();

            if ($certificateCount === 0) {
                // This vessel has 0 certificates, let's add some default ones
                Log::info("Fixing vessel with 0 certificates: {$vessel->name} (ID: {$vessel->id})");

                // Add default certificates (COR and DMLC)
                \App\Models\VesselCertificate::create([
                    'vessel_id' => $vessel->id,
                    'certificate_id' => 21, // COR certificate
                    'id_no' => 'DEFAULT-COR-' . $vessel->id,
                    'vessel_department' => 'FLAG',
                    'issued_date' => now(),
                    'expiry_date' => now()->addYears(5),
                ]);

                \App\Models\VesselCertificate::create([
                    'vessel_id' => $vessel->id,
                    'certificate_id' => 26, // DMLC certificate
                    'id_no' => 'DEFAULT-DMLC-' . $vessel->id,
                    'vessel_department' => 'FLAG',
                    'issued_date' => now(),
                    'expiry_date' => now()->addYears(5),
                ]);

                // Reload the vessel to get the new certificates
                $vessel->load('vesselCertificates.certificate');
            }
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('vessels.index')
                ->with('error', 'You do not have permission to create vessels.');
        }

        return view('vessels.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('vessels.index')
                ->with('error', 'You do not have permission to create vessels.');
        }

        $request->validate([
            'name' => 'required|string|max:100|unique:vessels',
        ]);

        // Start transaction
        DB::beginTransaction();

        try {
            // Create the vessel
            $vessel = Vessel::create([
                'name' => $request->name,
            ]);

            // Get all certificate types
            $certificates = Certificate::all();

            // Create empty certificates for all certificate types
            foreach ($certificates as $certificate) {
                \App\Models\VesselCertificate::create([
                    'vessel_id' => $vessel->id,
                    'certificate_id' => $certificate->id,
                ]);
            }

            // Add default data to COR and DMLC certificates
            $corCertificate = \App\Models\VesselCertificate::where('vessel_id', $vessel->id)
                ->where('certificate_id', 21) // COR certificate
                ->first();

            if ($corCertificate) {
                $corCertificate->update([
                    'id_no' => 'DEFAULT-COR-' . $vessel->id,
                    'vessel_department' => 'FLAG',
                    'issued_date' => now(),
                    'expiry_date' => now()->addYears(5),
                ]);
            }

            $dmlcCertificate = \App\Models\VesselCertificate::where('vessel_id', $vessel->id)
                ->where('certificate_id', 26) // DMLC certificate
                ->first();

            if ($dmlcCertificate) {
                $dmlcCertificate->update([
                    'id_no' => 'DEFAULT-DMLC-' . $vessel->id,
                    'vessel_department' => 'FLAG',
                    'issued_date' => now(),
                    'expiry_date' => now()->addYears(5),
                ]);
            }

            // Commit transaction
            DB::commit();

            return redirect()->route('vessels.show', $vessel)
                ->with('success', 'Vessel created successfully with all certificate types.');
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollback();
            Log::error("Vessel creation failed: " . $e->getMessage());

            return redirect()->route('vessels.index')
                ->with('error', 'An error occurred while creating the vessel.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Vessel $vessel)
    {
        // Check if vessel has 0 certificates and fix it
        $certificateCount = $vessel->vesselCertificates()->count();
        if ($certificateCount === 0) {
            // This vessel has 0 certificates, let's add all certificate types
            Log::info("Fixing vessel with 0 certificates in show method: {$vessel->name} (ID: {$vessel->id})");

            // Get all certificate types
            $certificates = Certificate::all();

            // Create empty certificates for all certificate types
            foreach ($certificates as $certificate) {
                \App\Models\VesselCertificate::create([
                    'vessel_id' => $vessel->id,
                    'certificate_id' => $certificate->id,
                ]);
            }

            // Add default data to COR and DMLC certificates
            $corCertificate = \App\Models\VesselCertificate::where('vessel_id', $vessel->id)
                ->where('certificate_id', 21) // COR certificate
                ->first();

            if ($corCertificate) {
                $corCertificate->update([
                    'id_no' => 'DEFAULT-COR-' . $vessel->id,
                    'vessel_department' => 'FLAG',
                    'issued_date' => now(),
                    'expiry_date' => now()->addYears(5),
                ]);
            }

            $dmlcCertificate = \App\Models\VesselCertificate::where('vessel_id', $vessel->id)
                ->where('certificate_id', 26) // DMLC certificate
                ->first();

            if ($dmlcCertificate) {
                $dmlcCertificate->update([
                    'id_no' => 'DEFAULT-DMLC-' . $vessel->id,
                    'vessel_department' => 'FLAG',
                    'issued_date' => now(),
                    'expiry_date' => now()->addYears(5),
                ]);
            }
        }

        // Load relationships with eager loading to ensure certificates are loaded
        $vessel->load([
            'vesselCertificates.certificate',
            'particulars',
            'vesselPlans.plan',
            'vesselSurveys.survey'
        ]);

        // Count only filled certificates (those with id_no, department, or dates)
        $filledCertificateCount = $vessel->vesselCertificates()
            ->where(function($query) {
                $query->whereNotNull('id_no')
                      ->orWhereNotNull('vessel_department')
                      ->orWhereNotNull('issued_date')
                      ->orWhereNotNull('expiry_date')
                      ->orWhereNotNull('attachment_path');
            })
            ->count();

        // Get all certificates, plans, and surveys for dropdowns
        $certificates = Certificate::orderBy('category')->orderBy('display_order')->get();
        $plans = Plan::orderBy('category')->orderBy('name')->get();
        $surveys = Survey::orderBy('category')->orderBy('name')->get();

        return view('vessels.show', compact('vessel', 'certificates', 'plans', 'surveys', 'filledCertificateCount'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Vessel $vessel)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('vessels.index')
                ->with('error', 'You do not have permission to edit vessels.');
        }

        return view('vessels.edit', compact('vessel'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Vessel $vessel)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('vessels.index')
                ->with('error', 'You do not have permission to edit vessels.');
        }

        $request->validate([
            'name' => 'required|string|max:100|unique:vessels,name,' . $vessel->id,
        ]);

        $vessel->update([
            'name' => $request->name,
        ]);

        return redirect()->route('vessels.show', $vessel)
            ->with('success', 'Vessel updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Vessel $vessel)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->route('vessels.index')
                ->with('error', 'You do not have permission to delete vessels.');
        }

        // Start transaction
        DB::beginTransaction();

        try {
            // Delete all related records
            $vessel->vesselCertificates()->delete();
            $vessel->vesselPlans()->delete();
            $vessel->vesselSurveys()->delete();

            if ($vessel->particulars) {
                $vessel->particulars->delete();
            }

            // Delete the vessel
            $vessel->delete();

            // Commit the transaction
            DB::commit();

            return redirect()->route('vessels.index')
                ->with('success', 'Vessel deleted successfully.');
        }
        catch (\Exception $e) {
            // Rollback the transaction on error
            DB::rollback();
            Log::error("Vessel deletion failed: " . $e->getMessage());

            return redirect()->route('vessels.index')
                ->with('error', 'An error occurred while deleting the vessel.');
        }
    }

    /**
     * Store a vessel certificate.
     */
    public function storeCertificate(Request $request)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to add certificates.');
        }

        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'certificate_id' => 'required|exists:certificates,id',
            'id_no' => 'nullable|string|max:100',
            'vessel_department' => 'nullable|string|max:100',
            'issued_date' => 'nullable|date',
            'expiry_date' => 'nullable|date|after_or_equal:issued_date',
        ]);

        // Handle multiple file uploads
        $attachmentPath = null;
        if ($request->hasFile('attachments')) {
            $paths = [];
            foreach ($request->file('attachments') as $file) {
                $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'certificates');
                $path = $file->storeAs('certificates', $fileName, 'public');
                $paths[] = $path;
            }
            $attachmentPath = !empty($paths) ? json_encode($paths) : null;
        }

        $vesselCertificate = \App\Models\VesselCertificate::create([
            'vessel_id' => $request->vessel_id,
            'certificate_id' => $request->certificate_id,
            'id_no' => $request->id_no,
            'vessel_department' => $request->vessel_department,
            'issued_date' => $request->issued_date,
            'expiry_date' => $request->expiry_date,
            'attachment_path' => $attachmentPath,
        ]);

        return redirect()->route('vessels.show', $request->vessel_id)
            ->with('success', 'Certificate added successfully.');
    }

    /**
     * Get certificate data for editing (JSON response).
     */
    public function editCertificate(\App\Models\VesselCertificate $vesselCertificate)
    {
        try {
            // Check if user has permission to manage certificates
            if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to edit certificates.'
                ], 403);
            }

            // Load the certificate with its relationships
            $vesselCertificate->load(['vessel', 'certificate']);

            // Format the data for the frontend
            $certificateData = [
                'id' => $vesselCertificate->id,
                'vessel_name' => $vesselCertificate->vessel->name,
                'certificate_name' => $vesselCertificate->certificate->name,
                'id_no' => $vesselCertificate->id_no,
                'vessel_department' => $vesselCertificate->vessel_department,
                'issued_date' => $vesselCertificate->issued_date ? $vesselCertificate->issued_date->format('Y-m-d') : null,
                'expiry_date' => $vesselCertificate->expiry_date ? $vesselCertificate->expiry_date->format('Y-m-d') : null,
                'attachment_path' => $vesselCertificate->attachment_path,
            ];

            return response()->json([
                'success' => true,
                'certificate' => $certificateData
            ]);

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Error fetching certificate data: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error fetching certificate data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a vessel certificate.
     */
    public function updateCertificate(Request $request, \App\Models\VesselCertificate $vesselCertificate)
    {
        try {
            // Check if user has permission to manage certificates
            if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
                if ($request->ajax() || $request->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'You do not have permission to update certificates.'
                    ], 403);
                }

                return redirect()->back()
                    ->with('error', 'You do not have permission to update certificates.');
            }

            $request->validate([
                'id_no' => 'nullable|string|max:100',
                'vessel_department' => 'nullable|string|max:100',
                'issued_date' => 'nullable|date',
                'expiry_date' => 'nullable|date|after_or_equal:issued_date',
            ]);

            // Handle file uploads (both single and multiple)
            if ($request->hasFile('attachments') || $request->hasFile('attachment')) {
                try {
                    // Start with existing attachments - we'll ADD new ones
                    $attachments = [];

                    // Get existing attachments first
                    if ($vesselCertificate->attachment_path) {
                        if (is_string($vesselCertificate->attachment_path) && !is_array(json_decode($vesselCertificate->attachment_path, true))) {
                            // Single attachment as string
                            $attachments = [$vesselCertificate->attachment_path];
                        } else {
                            // Multiple attachments as array
                            $existingAttachments = is_array($vesselCertificate->attachment_path)
                                ? $vesselCertificate->attachment_path
                                : (json_decode($vesselCertificate->attachment_path, true) ?: []);
                            $attachments = $existingAttachments;
                        }
                    }

                    // Handle multiple files (from vessel detail page) - ADD to existing
                    if ($request->hasFile('attachments')) {
                        foreach ($request->file('attachments') as $file) {
                            // Check file size
                            if ($file->getSize() > 32 * 1024 * 1024) { // 32MB limit
                                throw new \Exception('File size exceeds the 32MB limit: ' . $file->getClientOriginalName());
                            }

                            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'certificates');
                            $path = $file->storeAs('certificates', $fileName, 'public');
                            $attachments[] = $path; // ADD to existing attachments
                        }
                    }

                    // Handle single file (from dashboard modal) - ADD to existing
                    if ($request->hasFile('attachment')) {
                        $file = $request->file('attachment');

                        // Check file size
                        if ($file->getSize() > 32 * 1024 * 1024) { // 32MB limit
                            throw new \Exception('File size exceeds the 32MB limit: ' . $file->getClientOriginalName());
                        }

                        // ADD to existing attachments (not replace)
                        $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'certificates');
                        $path = $file->storeAs('certificates', $fileName, 'public');
                        $attachments[] = $path; // ADD to existing attachments
                    }

                    // No need to delete old attachments since we're adding to them

                    $vesselCertificate->attachment_path = json_encode($attachments);
                } catch (\Exception $e) {
                    \Illuminate\Support\Facades\Log::error('Error uploading attachments: ' . $e->getMessage());

                    if ($request->ajax() || $request->wantsJson()) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Error uploading attachments: ' . $e->getMessage()
                        ], 422);
                    }

                    return redirect()->back()
                        ->with('error', 'Error uploading attachments: ' . $e->getMessage());
                }
            }

            // Update certificate information
            $vesselCertificate->id_no = $request->id_no;
            $vesselCertificate->vessel_department = $request->vessel_department;
            $vesselCertificate->issued_date = $request->issued_date;
            $vesselCertificate->expiry_date = $request->expiry_date;
            $vesselCertificate->save();

            // Check if it's an AJAX request
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Certificate updated successfully.',
                    'redirect' => route('vessels.show', $vesselCertificate->vessel_id)
                ]);
            }

            return redirect()->route('vessels.show', $vesselCertificate->vessel_id)
                ->with('success', 'Certificate updated successfully.');
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Error updating certificate: ' . $e->getMessage());

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error updating certificate: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Error updating certificate: ' . $e->getMessage());
        }
    }

    /**
     * Remove a vessel certificate.
     */
    public function destroyCertificate(\App\Models\VesselCertificate $vesselCertificate)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to delete certificates.'
                ], 403);
            }

            return redirect()->back()
                ->with('error', 'You do not have permission to delete certificates.');
        }

        try {
            $vesselId = $vesselCertificate->vessel_id;
            $vesselName = $vesselCertificate->vessel->name;
            $certificateName = $vesselCertificate->certificate->name;

            // Delete attachments from storage
            if ($vesselCertificate->attachment_path) {
                // Get the CertificateController to use its deleteFile method
                $certificateController = app()->make(\App\Http\Controllers\CertificateController::class);

                // Handle both string and array formats
                if (is_string($vesselCertificate->attachment_path) && !is_array(json_decode($vesselCertificate->attachment_path, true))) {
                    // Single attachment as string
                    $certificateController->deleteFile($vesselCertificate->attachment_path);
                } else {
                    // Multiple attachments as array
                    $attachments = is_array($vesselCertificate->attachment_path)
                        ? $vesselCertificate->attachment_path
                        : json_decode($vesselCertificate->attachment_path, true);

                    if (is_array($attachments)) {
                        foreach ($attachments as $attachment) {
                            $certificateController->deleteFile($attachment);
                        }
                    }
                }
            }

            $vesselCertificate->delete();

            // Handle AJAX requests
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => "Deleted {$certificateName} for {$vesselName}"
                ]);
            }

            return redirect()->route('vessels.show', $vesselId)
                ->with('success', 'Certificate deleted successfully.');
        } catch (\Exception $e) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete certificate: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Failed to delete certificate: ' . $e->getMessage());
        }
    }

    /**
     * Store a vessel plan.
     */
    public function storePlan(Request $request)
    {
        // Check if user has permission to manage plans
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_plans')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to add plans.');
        }

        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'plan_id' => 'required|exists:plans,id',
            'attachment' => 'nullable|file|max:32768', // 32MB
        ]);

        // Handle file upload
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'plans');
            $attachmentPath = $file->storeAs('plans', $fileName, 'public');
        }

        $vesselPlan = \App\Models\VesselPlan::create([
            'vessel_id' => $request->vessel_id,
            'plan_id' => $request->plan_id,
            'attachment_path' => $attachmentPath,
        ]);

        return redirect()->route('vessels.show', $request->vessel_id)
            ->with('success', 'Plan added successfully.');
    }

    /**
     * Update a vessel plan.
     */
    public function updatePlan(Request $request, \App\Models\VesselPlan $vesselPlan)
    {
        // Check if user has permission to manage plans
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_plans')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to update plans.');
        }

        $request->validate([
            'attachment' => 'nullable|file|max:10240',
        ]);

        // Handle file upload
        if ($request->hasFile('attachment')) {
            // Delete old attachment
            if ($vesselPlan->attachment_path) {
                Storage::disk('public')->delete($vesselPlan->attachment_path);
            }

            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'plans');
            $vesselPlan->attachment_path = $file->storeAs('plans', $fileName, 'public');
            $vesselPlan->save();
        }

        return redirect()->route('vessels.show', $vesselPlan->vessel_id)
            ->with('success', 'Plan updated successfully.');
    }

    /**
     * Remove a vessel plan.
     */
    public function destroyPlan(\App\Models\VesselPlan $vesselPlan)
    {
        // Check if user has permission to manage plans
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_plans')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to delete plans.');
        }

        $vesselId = $vesselPlan->vessel_id;

        // Delete attachment from storage
        if ($vesselPlan->attachment_path) {
            Storage::disk('public')->delete($vesselPlan->attachment_path);
        }

        $vesselPlan->delete();

        return redirect()->route('vessels.show', $vesselId)
            ->with('success', 'Plan deleted successfully.');
    }

    /**
     * Store a vessel survey.
     */
    public function storeSurvey(Request $request)
    {
        // Check if user has permission to manage surveys
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_surveys')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to add surveys.');
        }

        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'survey_id' => 'required|exists:surveys,id',
            'last_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'window_date' => 'nullable|date',
            'status' => 'nullable|string|max:50',
            'general_note' => 'nullable|string',
        ]);

        $vesselSurvey = \App\Models\VesselSurvey::create([
            'vessel_id' => $request->vessel_id,
            'survey_id' => $request->survey_id,
            'last_date' => $request->last_date,
            'due_date' => $request->due_date,
            'window_date' => $request->window_date,
            'status' => $request->status,
            'general_note' => $request->general_note,
        ]);

        return redirect()->route('vessels.show', $request->vessel_id)
            ->with('success', 'Survey added successfully.');
    }

    /**
     * Update a vessel survey.
     */
    public function updateSurvey(Request $request, \App\Models\VesselSurvey $vesselSurvey)
    {
        // Check if user has permission to manage surveys
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_surveys')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to update surveys.');
        }

        $request->validate([
            'last_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'window_date' => 'nullable|date',
            'status' => 'nullable|string|max:50',
            'general_note' => 'nullable|string',
        ]);

        $vesselSurvey->update([
            'last_date' => $request->last_date,
            'due_date' => $request->due_date,
            'window_date' => $request->window_date,
            'status' => $request->status,
            'general_note' => $request->general_note,
        ]);

        return redirect()->route('vessels.show', $vesselSurvey->vessel_id)
            ->with('success', 'Survey updated successfully.');
    }

    /**
     * Remove a vessel survey.
     */
    public function destroySurvey(\App\Models\VesselSurvey $vesselSurvey)
    {
        // Check if user has permission to manage surveys
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_surveys')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to delete surveys.');
        }

        $vesselId = $vesselSurvey->vessel_id;
        $vesselSurvey->delete();

        return redirect()->route('vessels.show', $vesselId)
            ->with('success', 'Survey deleted successfully.');
    }

    /**
     * Store vessel particulars.
     */
    public function storeParticulars(Request $request)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to add particulars.');
        }

        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'imo_number' => 'nullable|string|max:100',
            'attachment' => 'nullable|file|max:32768', // 32MB
        ]);

        // Check if particulars already exist
        $vessel = Vessel::findOrFail($request->vessel_id);

        // Handle file upload
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'particulars');
            $attachmentPath = $file->storeAs('particulars', $fileName, 'public');
        }

        // Create or update particulars
        if ($vessel->particulars) {
            // Update existing particulars
            $vessel->particulars->update([
                'imo_number' => $request->imo_number,
                'attachment_path' => $attachmentPath ?? $vessel->particulars->attachment_path,
            ]);
        } else {
            // Create new particulars
            \App\Models\VesselParticular::create([
                'vessel_id' => $request->vessel_id,
                'imo_number' => $request->imo_number,
                'attachment_path' => $attachmentPath,
            ]);
        }

        return redirect()->route('vessels.show', $request->vessel_id)
            ->with('success', 'Vessel particulars added successfully.');
    }

    /**
     * Update vessel particulars.
     */
    public function updateParticulars(Request $request, \App\Models\VesselParticular $vesselParticular)
    {
        // Check if user has permission to manage certificates
        if (!auth()->user()->is_admin && !auth()->user()->hasPermission('manage_certificates')) {
            return redirect()->back()
                ->with('error', 'You do not have permission to update particulars.');
        }

        $request->validate([
            'imo_number' => 'nullable|string|max:100',
            'attachment' => 'nullable|file|max:32768', // 32MB
        ]);

        // Handle file upload
        if ($request->hasFile('attachment')) {
            // Delete old attachment if it exists
            if ($vesselParticular->attachment_path) {
                Storage::disk('public')->delete($vesselParticular->attachment_path);
            }

            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'particulars');
            $vesselParticular->attachment_path = $file->storeAs('particulars', $fileName, 'public');
        }

        $vesselParticular->imo_number = $request->imo_number;
        $vesselParticular->save();

        return redirect()->route('vessels.show', $vesselParticular->vessel_id)
            ->with('success', 'Vessel particulars updated successfully.');
    }

    /**
     * View or download a certificate file.
     */
    public function viewCertificateFile(Request $request)
    {
        $request->validate([
            'path' => 'required|string',
        ]);

        $path = $request->path;

        // Handle different path formats
        if (strpos($path, 'http') === 0) {
            // External URL - redirect to it
            return redirect($path);
        }

        // Clean up the path
        $path = str_replace('\\', '', $path);

        // Remove any JSON formatting if present
        if (substr($path, 0, 2) === '["' && substr($path, -2) === '"]') {
            $path = substr($path, 2, -2);
        }

        // Handle old path format
        if (strpos($path, '/certificates/uploads/') === 0) {
            // Old path format - try to find in storage/app/public/certificates
            $cleanPath = str_replace('/certificates/uploads/', 'certificates/', $path);

            if (Storage::disk('public')->exists($cleanPath)) {
                return Storage::disk('public')->download($cleanPath);
            }

            // If not found, try the original path
            if (file_exists(public_path('storage' . $path))) {
                return response()->file(public_path('storage' . $path));
            }
        } else {
            // New path format
            if (Storage::disk('public')->exists($path)) {
                return Storage::disk('public')->download($path);
            }
        }

        // If file not found
        return response()->json(['error' => 'File not found'], 404);
    }

    /**
     * Preview a certificate file in the browser.
     */
    public function previewCertificateFile(Request $request)
    {
        $request->validate([
            'path' => 'required|string',
        ]);

        $path = $request->path;

        // Handle different path formats
        if (strpos($path, 'http') === 0) {
            // External URL - redirect to it
            return redirect($path);
        }

        // Clean up the path
        $path = str_replace('\\', '', $path);

        // Remove any JSON formatting if present
        if (substr($path, 0, 2) === '["' && substr($path, -2) === '"]') {
            $path = substr($path, 2, -2);
        }

        // Get file extension
        $extension = pathinfo($path, PATHINFO_EXTENSION);

        // Handle old path format
        if (strpos($path, '/certificates/uploads/') === 0) {
            // Old path format - try to find in storage/app/public/certificates
            $cleanPath = str_replace('/certificates/uploads/', 'certificates/', $path);

            if (Storage::disk('public')->exists($cleanPath)) {
                $mimeType = Storage::disk('public')->mimeType($cleanPath);
                return response()->file(storage_path('app/public/' . $cleanPath), [
                    'Content-Type' => $mimeType,
                    'Content-Disposition' => 'inline; filename="' . basename($cleanPath) . '"'
                ]);
            }

            // If not found, try the original path
            if (file_exists(public_path('storage' . $path))) {
                $mimeType = mime_content_type(public_path('storage' . $path));
                return response()->file(public_path('storage' . $path), [
                    'Content-Type' => $mimeType,
                    'Content-Disposition' => 'inline; filename="' . basename($path) . '"'
                ]);
            }
        } else {
            // New path format
            if (Storage::disk('public')->exists($path)) {
                $mimeType = Storage::disk('public')->mimeType($path);
                return response()->file(storage_path('app/public/' . $path), [
                    'Content-Type' => $mimeType,
                    'Content-Disposition' => 'inline; filename="' . basename($path) . '"'
                ]);
            }
        }

        // If file not found
        return response()->json(['error' => 'File not found'], 404);
    }

    /**
     * Generate a unique file name by checking for existing files and adding _1, _2, etc.
     */
    private function generateUniqueFileName($originalName, $directory)
    {
        $fileName = $originalName;
        $counter = 1;

        // Keep checking until we find a unique name
        while (\Illuminate\Support\Facades\Storage::disk('public')->exists($directory . '/' . $fileName)) {
            $pathInfo = pathinfo($originalName);
            $name = $pathInfo['filename'];
            $extension = isset($pathInfo['extension']) ? '.' . $pathInfo['extension'] : '';

            $fileName = $name . '_' . $counter . $extension;
            $counter++;
        }

        return $fileName;
    }
}
