<?php

namespace App\Http\Controllers;

use App\Models\Vessel;
use App\Models\VesselParticular;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class VesselParticularsController extends Controller
{
    /**
     * Store a newly created vessel particular in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'imo_number' => 'nullable|string|max:100',
            'attachment' => 'nullable|file|max:32768', // 32MB
        ]);

        // Check if vessel already has particulars
        $existingParticulars = VesselParticular::where('vessel_id', $request->vessel_id)->first();
        if ($existingParticulars) {
            return redirect()->back()
                ->with('error', 'This vessel already has particulars. Please edit the existing particulars instead.');
        }

        $vesselParticular = new VesselParticular();
        $vesselParticular->vessel_id = $request->vessel_id;
        $vesselParticular->imo_number = $request->imo_number;

        // Handle file upload
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'particulars');
            $path = $file->storeAs('particulars', $fileName, 'public');
            $vesselParticular->attachment_path = $path;
        }

        $vesselParticular->save();

        return redirect()->route('vessels.show', $request->vessel_id)
            ->with('success', 'Vessel particulars added successfully.');
    }

    /**
     * Update the specified vessel particular in storage.
     */
    public function update(Request $request, VesselParticular $vesselParticular)
    {
        $request->validate([
            'imo_number' => 'nullable|string|max:100',
            'attachment' => 'nullable|file|max:32768', // 32MB
        ]);

        $vesselParticular->imo_number = $request->imo_number;

        // Handle file upload
        if ($request->hasFile('attachment')) {
            // Delete old file if exists
            if ($vesselParticular->attachment_path) {
                Storage::disk('public')->delete($vesselParticular->attachment_path);
            }

            $file = $request->file('attachment');
            $fileName = $this->generateUniqueFileName($file->getClientOriginalName(), 'particulars');
            $path = $file->storeAs('particulars', $fileName, 'public');
            $vesselParticular->attachment_path = $path;
        }

        $vesselParticular->save();

        return redirect()->route('vessels.show', $vesselParticular->vessel_id)
            ->with('success', 'Vessel particulars updated successfully.');
    }

    /**
     * Remove the specified vessel particular from storage.
     */
    public function destroy(VesselParticular $vesselParticular)
    {
        // Delete attachment if exists
        if ($vesselParticular->attachment_path) {
            Storage::disk('public')->delete($vesselParticular->attachment_path);
        }

        $vesselId = $vesselParticular->vessel_id;
        $vesselParticular->delete();

        return redirect()->route('vessels.show', $vesselId)
            ->with('success', 'Vessel particulars deleted successfully.');
    }

    /**
     * Generate a unique file name by checking for existing files and adding _1, _2, etc.
     */
    private function generateUniqueFileName($originalName, $directory)
    {
        $fileName = $originalName;
        $counter = 1;

        // Keep checking until we find a unique name
        while (\Illuminate\Support\Facades\Storage::disk('public')->exists($directory . '/' . $fileName)) {
            $pathInfo = pathinfo($originalName);
            $name = $pathInfo['filename'];
            $extension = isset($pathInfo['extension']) ? '.' . $pathInfo['extension'] : '';

            $fileName = $name . '_' . $counter . $extension;
            $counter++;
        }

        return $fileName;
    }
}
