<?php
// This script imports all certificates from the aikhrrte_certificates.sql file
// and creates empty certificates for all certificate types for all vessels

// Set up Laravel environment
require __DIR__ . '/vendor/autoload.php';
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\Vessel;
use App\Models\Certificate;
use App\Models\VesselCertificate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// Function to parse SQL file and extract vessel certificates
function extractVesselCertificatesFromSql($sqlFilePath) {
    if (!file_exists($sqlFilePath)) {
        echo "SQL file not found at: $sqlFilePath\n";
        return [];
    }

    $sqlContent = file_get_contents($sqlFilePath);

    // Extract the INSERT statements for vessel_certificates
    preg_match_all('/INSERT INTO `vessel_certificates`.*VALUES\s*(.*?);/s', $sqlContent, $matches);

    $certificates = [];

    foreach ($matches[1] as $valuesString) {
        // Parse the values
        preg_match_all('/\(([^)]+)\)/', $valuesString, $valueMatches);

        foreach ($valueMatches[1] as $valueString) {
            // Split by comma, but respect quoted strings
            $fields = [];
            $inQuote = false;
            $currentField = '';

            for ($i = 0; $i < strlen($valueString); $i++) {
                $char = $valueString[$i];

                if ($char === "'" && ($i === 0 || $valueString[$i-1] !== '\\')) {
                    $inQuote = !$inQuote;
                    $currentField .= $char;
                } elseif ($char === ',' && !$inQuote) {
                    $fields[] = trim($currentField);
                    $currentField = '';
                } else {
                    $currentField .= $char;
                }
            }

            // Add the last field
            if (!empty($currentField)) {
                $fields[] = trim($currentField);
            }

            // Skip if not enough fields
            if (count($fields) < 10) {
                echo "Skipping row with insufficient fields: " . count($fields) . "\n";
                continue;
            }

            // Clean up fields
            $id = (int)trim($fields[0]);
            $vesselId = (int)trim($fields[1]);
            $certificateId = (int)trim($fields[2]);

            $idNo = trim($fields[3], "'");
            if ($idNo === 'NULL' || $idNo === 'null') $idNo = null;

            $vesselDepartment = trim($fields[4], "'");
            if ($vesselDepartment === 'NULL' || $vesselDepartment === 'null') $vesselDepartment = null;

            $issuedDate = trim($fields[5], "'");
            if ($issuedDate === 'NULL' || $issuedDate === 'null' || $issuedDate === '0000-00-00') $issuedDate = null;

            $expiryDate = trim($fields[6], "'");
            if ($expiryDate === 'NULL' || $expiryDate === 'null' || $expiryDate === '0000-00-00') $expiryDate = null;

            $attachmentPath = trim($fields[7], "'");
            if ($attachmentPath === 'NULL' || $attachmentPath === 'null') $attachmentPath = null;

            $certificates[] = [
                'id' => $id,
                'vessel_id' => $vesselId,
                'certificate_id' => $certificateId,
                'id_no' => $idNo,
                'vessel_department' => $vesselDepartment,
                'issued_date' => $issuedDate,
                'expiry_date' => $expiryDate,
                'attachment_path' => $attachmentPath,
            ];
        }
    }

    return $certificates;
}

// Main execution
echo "Starting certificate import from SQL file...\n";

// Get all vessels
$vessels = Vessel::all();
echo "Found " . $vessels->count() . " vessels\n";

// Get all certificates
$certificates = Certificate::all();
echo "Found " . $certificates->count() . " certificate types\n";

// Extract certificates from SQL file
$sqlFilePath = __DIR__ . '/aikhrrte_certificates.sql';
$sqlCertificates = extractVesselCertificatesFromSql($sqlFilePath);
echo "Extracted " . count($sqlCertificates) . " certificates from SQL file\n";

// Group certificates by vessel
$certificatesByVessel = [];
foreach ($sqlCertificates as $cert) {
    $vesselId = $cert['vessel_id'];
    if (!isset($certificatesByVessel[$vesselId])) {
        $certificatesByVessel[$vesselId] = [];
    }
    $certificatesByVessel[$vesselId][] = $cert;
}

// Begin transaction
DB::beginTransaction();

try {
    $importCount = 0;
    $skipCount = 0;
    $updateCount = 0;

    foreach ($vessels as $vessel) {
        $vesselId = $vessel->id;
        $existingCount = VesselCertificate::where('vessel_id', $vesselId)->count();
        $sqlCount = isset($certificatesByVessel[$vesselId]) ? count($certificatesByVessel[$vesselId]) : 0;

        echo "Vessel: {$vessel->name} (ID: {$vesselId}) - Existing: {$existingCount}, SQL: {$sqlCount}\n";

        // Create empty certificates for all certificate types
        foreach ($certificates as $certificate) {
            $exists = VesselCertificate::where('vessel_id', $vesselId)
                ->where('certificate_id', $certificate->id)
                ->exists();

            if (!$exists) {
                VesselCertificate::create([
                    'vessel_id' => $vesselId,
                    'certificate_id' => $certificate->id,
                ]);

                $importCount++;
                echo "  Created empty certificate for {$certificate->name}\n";
            }
        }

        // Import certificates from SQL file
        if (isset($certificatesByVessel[$vesselId])) {
            foreach ($certificatesByVessel[$vesselId] as $cert) {
                // Find the certificate
                $existingCert = VesselCertificate::where('vessel_id', $vesselId)
                    ->where('certificate_id', $cert['certificate_id'])
                    ->first();

                if ($existingCert) {
                    // Update existing certificate if it's empty
                    $isEmpty = !$existingCert->id_no && !$existingCert->vessel_department &&
                              !$existingCert->issued_date && !$existingCert->expiry_date;

                    if ($isEmpty || true) { // Always update to ensure we get all data
                        $existingCert->id_no = $cert['id_no'] ?? $existingCert->id_no;
                        $existingCert->vessel_department = $cert['vessel_department'] ?? $existingCert->vessel_department;
                        $existingCert->issued_date = $cert['issued_date'] ?? $existingCert->issued_date;
                        $existingCert->expiry_date = $cert['expiry_date'] ?? $existingCert->expiry_date;
                        $existingCert->attachment_path = $cert['attachment_path'] ?? $existingCert->attachment_path;
                        $existingCert->save();

                        $updateCount++;
                        echo "  Updated certificate ID: {$existingCert->id}, Type: {$cert['certificate_id']}\n";
                    } else {
                        $skipCount++;
                    }
                }
            }
        }
    }

    // Commit transaction
    DB::commit();

    echo "Import completed successfully!\n";
    echo "Created: {$importCount}, Updated: {$updateCount}, Skipped: {$skipCount}\n";

} catch (\Exception $e) {
    // Rollback transaction on error
    DB::rollBack();

    echo "Error during import: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}
