<?php
// This script directly imports vessel particulars and surveys

// Load the Laravel environment
require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Models\Vessel;
use App\Models\Survey;
use App\Models\VesselSurvey;
use App\Models\VesselParticular;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

// Main execution
echo "Starting direct import of vessel particulars and surveys...\n";

// Get all vessels
$vessels = Vessel::all();
echo "Found " . $vessels->count() . " vessels\n";

// Get all surveys
$surveys = Survey::all();
echo "Found " . $surveys->count() . " survey types\n";

// Begin transaction
DB::beginTransaction();

try {
    // Import vessel particulars
    $particularImportCount = 0;
    $particularUpdateCount = 0;
    $particularSkipCount = 0;
    
    // Particulars data from SQL file
    $particularsData = [
        ['id' => 2, 'vessel_id' => 1, 'imo_number' => '9053830', 'attachment_path' => null],
        ['id' => 4, 'vessel_id' => 3, 'imo_number' => '9218208', 'attachment_path' => 'uploads/particulars/67f4d3c0671b9.pdf'],
        ['id' => 10, 'vessel_id' => 16, 'imo_number' => '9306299', 'attachment_path' => 'uploads/particulars/67f4d46a7999a.pdf'],
        ['id' => 11, 'vessel_id' => 6, 'imo_number' => '9008134', 'attachment_path' => 'uploads/particulars/67f4d46659d45.pdf'],
        ['id' => 12, 'vessel_id' => 15, 'imo_number' => '9300855', 'attachment_path' => 'uploads/particulars/67f4d4606905a.pdf'],
        ['id' => 13, 'vessel_id' => 8, 'imo_number' => '9331804', 'attachment_path' => 'uploads/particulars/67f4d3a54a26f.pdf'],
        ['id' => 14, 'vessel_id' => 17, 'imo_number' => '9180384', 'attachment_path' => 'uploads/particulars/67f4d3b422ca6.pdf'],
        ['id' => 15, 'vessel_id' => 9, 'imo_number' => '9205615', 'attachment_path' => 'uploads/particulars/67f4d3ca54053.pdf'],
        ['id' => 16, 'vessel_id' => 11, 'imo_number' => '9217826', 'attachment_path' => 'uploads/particulars/67f4d43e327d7.pdf'],
        ['id' => 17, 'vessel_id' => 12, 'imo_number' => '9338125', 'attachment_path' => 'uploads/particulars/67f4d4450dd08.pdf'],
        ['id' => 18, 'vessel_id' => 13, 'imo_number' => '9793557', 'attachment_path' => 'uploads/particulars/67f4d44c08a11.pdf'],
        ['id' => 19, 'vessel_id' => 14, 'imo_number' => '9300831', 'attachment_path' => 'uploads/particulars/67f4d45692081.pdf'],
        ['id' => 20, 'vessel_id' => 7, 'imo_number' => '9306421', 'attachment_path' => 'uploads/particulars/67f4d35b7a26b.pdf'],
        ['id' => 23, 'vessel_id' => 10, 'imo_number' => '9362657', 'attachment_path' => 'uploads/particulars/67f4d43762cd2.pdf'],
    ];
    
    foreach ($particularsData as $particular) {
        $vesselId = $particular['vessel_id'];
        
        // Check if vessel exists
        $vessel = Vessel::find($vesselId);
        if (!$vessel) {
            echo "  Vessel ID {$vesselId} not found, skipping particular\n";
            $particularSkipCount++;
            continue;
        }
        
        // Check if particular already exists
        $existingParticular = VesselParticular::where('vessel_id', $vesselId)->first();
        
        if ($existingParticular) {
            // Update existing particular
            $existingParticular->imo_number = $particular['imo_number'];
            
            // Only update attachment_path if it's not null
            if ($particular['attachment_path']) {
                $existingParticular->attachment_path = $particular['attachment_path'];
            }
            
            $existingParticular->save();
            
            $particularUpdateCount++;
            echo "  Updated particular for vessel ID {$vesselId}\n";
        } else {
            // Create new particular
            VesselParticular::create([
                'vessel_id' => $vesselId,
                'imo_number' => $particular['imo_number'],
                'attachment_path' => $particular['attachment_path'],
            ]);
            
            $particularImportCount++;
            echo "  Created particular for vessel ID {$vesselId}\n";
        }
    }
    
    // Import a sample of vessel surveys (first 20 from the SQL file)
    $surveyImportCount = 0;
    $surveyUpdateCount = 0;
    $surveySkipCount = 0;
    
    // Sample of surveys data from SQL file
    $surveysData = [
        ['id' => 6, 'vessel_id' => 12, 'survey_id' => 1, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 7, 'vessel_id' => 12, 'survey_id' => 2, 'last_date' => null, 'due_date' => '2026-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 8, 'vessel_id' => 12, 'survey_id' => 3, 'last_date' => null, 'due_date' => '2027-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 9, 'vessel_id' => 12, 'survey_id' => 5, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 10, 'vessel_id' => 12, 'survey_id' => 6, 'last_date' => null, 'due_date' => '2026-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 11, 'vessel_id' => 12, 'survey_id' => 7, 'last_date' => null, 'due_date' => '2024-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 12, 'vessel_id' => 12, 'survey_id' => 8, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 13, 'vessel_id' => 12, 'survey_id' => 11, 'last_date' => null, 'due_date' => '2027-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 14, 'vessel_id' => 12, 'survey_id' => 12, 'last_date' => null, 'due_date' => '2026-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 15, 'vessel_id' => 12, 'survey_id' => 13, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 16, 'vessel_id' => 12, 'survey_id' => 14, 'last_date' => null, 'due_date' => '2027-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 17, 'vessel_id' => 12, 'survey_id' => 15, 'last_date' => null, 'due_date' => '2026-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 18, 'vessel_id' => 12, 'survey_id' => 16, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 19, 'vessel_id' => 12, 'survey_id' => 17, 'last_date' => null, 'due_date' => '2027-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 20, 'vessel_id' => 12, 'survey_id' => 18, 'last_date' => null, 'due_date' => '2026-03-17', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 21, 'vessel_id' => 12, 'survey_id' => 19, 'last_date' => null, 'due_date' => '2027-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 22, 'vessel_id' => 12, 'survey_id' => 20, 'last_date' => null, 'due_date' => '2026-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 23, 'vessel_id' => 12, 'survey_id' => 21, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 24, 'vessel_id' => 12, 'survey_id' => 22, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 25, 'vessel_id' => 12, 'survey_id' => 23, 'last_date' => null, 'due_date' => '2027-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 26, 'vessel_id' => 12, 'survey_id' => 24, 'last_date' => null, 'due_date' => '2026-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
        ['id' => 27, 'vessel_id' => 12, 'survey_id' => 25, 'last_date' => null, 'due_date' => '2025-08-12', 'window_date' => null, 'status' => null, 'general_note' => null],
    ];
    
    foreach ($surveysData as $survey) {
        $vesselId = $survey['vessel_id'];
        $surveyId = $survey['survey_id'];
        
        // Check if vessel exists
        $vessel = Vessel::find($vesselId);
        if (!$vessel) {
            echo "  Vessel ID {$vesselId} not found, skipping survey\n";
            $surveySkipCount++;
            continue;
        }
        
        // Check if survey type exists
        $surveyType = Survey::find($surveyId);
        if (!$surveyType) {
            echo "  Survey ID {$surveyId} not found, skipping survey\n";
            $surveySkipCount++;
            continue;
        }
        
        // Check if survey already exists
        $existingSurvey = VesselSurvey::where('vessel_id', $vesselId)
            ->where('survey_id', $surveyId)
            ->first();
        
        if ($existingSurvey) {
            // Update existing survey
            $existingSurvey->last_date = $survey['last_date'];
            $existingSurvey->due_date = $survey['due_date'];
            $existingSurvey->window_date = $survey['window_date'];
            $existingSurvey->status = $survey['status'];
            $existingSurvey->general_note = $survey['general_note'];
            $existingSurvey->save();
            
            $surveyUpdateCount++;
            echo "  Updated survey for vessel ID {$vesselId}, survey ID {$surveyId}\n";
        } else {
            // Create new survey
            VesselSurvey::create([
                'vessel_id' => $vesselId,
                'survey_id' => $surveyId,
                'last_date' => $survey['last_date'],
                'due_date' => $survey['due_date'],
                'window_date' => $survey['window_date'],
                'status' => $survey['status'],
                'general_note' => $survey['general_note'],
            ]);
            
            $surveyImportCount++;
            echo "  Created survey for vessel ID {$vesselId}, survey ID {$surveyId}\n";
        }
    }
    
    // Commit transaction
    DB::commit();
    
    echo "Import completed successfully!\n";
    echo "Particulars: Created: {$particularImportCount}, Updated: {$particularUpdateCount}, Skipped: {$particularSkipCount}\n";
    echo "Surveys: Created: {$surveyImportCount}, Updated: {$surveyUpdateCount}, Skipped: {$surveySkipCount}\n";
    
} catch (\Exception $e) {
    // Rollback transaction on error
    DB::rollBack();
    
    echo "Error during import: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}
