<?php
/**
 * Direct script to delete certificate attachments
 *
 * This script bypasses the normal Laravel flow to directly delete files and update the database
 */

// Set error reporting to log errors but not display them
error_reporting(E_ALL);
ini_set('display_errors', 0);

// Always set the content type to JSON
header('Content-Type: application/json');

// Function to return a JSON error response
function returnJsonError($message, $code = 500) {
    echo json_encode([
        'success' => false,
        'message' => $message
    ]);
    exit;
}

// Wrap everything in a try-catch to ensure we always return valid JSON
try {
    // Load the Laravel environment
    require __DIR__.'/vendor/autoload.php';
    $app = require_once __DIR__.'/bootstrap/app.php';
    $kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
    $kernel->bootstrap();

    // Check if user is authenticated
    if (!auth()->check()) {
        returnJsonError('You must be logged in to perform this action.');
    }

    // Check CSRF token
    if (isset($_SERVER['HTTP_X_CSRF_TOKEN'])) {
        // If token is provided in header
        $token = $_SERVER['HTTP_X_CSRF_TOKEN'];
    } elseif (isset($_POST['_token'])) {
        // If token is provided in form
        $token = $_POST['_token'];
    } else {
        // No token provided
        returnJsonError('CSRF token missing');
    }

    // Verify the token
    if (!hash_equals(csrf_token(), $token)) {
        returnJsonError('CSRF token mismatch');
    }

// Get the request parameters
$vesselCertificateId = $_POST['vessel_certificate_id'] ?? null;
$attachmentIndex = $_POST['attachment_index'] ?? null;

if (!$vesselCertificateId || !isset($attachmentIndex)) {
    die(json_encode([
        'success' => false,
        'message' => 'Missing required parameters'
    ]));
}

// Find the vessel certificate
$vesselCertificate = \App\Models\VesselCertificate::find($vesselCertificateId);
if (!$vesselCertificate) {
    die(json_encode([
        'success' => false,
        'message' => 'Certificate not found'
    ]));
}

// Log the current attachment path
$originalPath = $vesselCertificate->attachment_path;
// Don't output anything before the final JSON response
// We'll store logs in an array instead
$logs = [];
$logs[] = "Original attachment path: " . json_encode($originalPath);

// Function to aggressively delete a file
function deleteFileAggressively($path, &$logs) {
    $logs[] = "Attempting to delete: $path";

    // Try all possible storage locations
    $possiblePaths = [
        // Storage paths
        storage_path('app/public/' . $path),
        storage_path('app/public/certificates/' . basename($path)),
        storage_path('app/public/' . ltrim($path, '/')),

        // Public paths
        public_path('storage/' . $path),
        public_path('storage/certificates/' . basename($path)),
        public_path('storage/' . ltrim($path, '/')),
        public_path($path),

        // Absolute paths
        $path,
        '/var/www/html/storage/app/public/' . $path,
        '/var/www/html/public/storage/' . $path,
    ];

    $deleted = false;
    foreach ($possiblePaths as $fullPath) {
        if (file_exists($fullPath)) {
            $logs[] = "Found file at: $fullPath";
            if (unlink($fullPath)) {
                $logs[] = "Successfully deleted: $fullPath";
                $deleted = true;
            } else {
                $logs[] = "Failed to delete: $fullPath";
            }
        }
    }

    // Also try Laravel's Storage facade
    try {
        if (\Illuminate\Support\Facades\Storage::disk('public')->exists($path)) {
            \Illuminate\Support\Facades\Storage::disk('public')->delete($path);
            $logs[] = "Deleted via Storage facade: $path";
            $deleted = true;
        }

        // Try with certificates prefix
        $certPath = 'certificates/' . basename($path);
        if (\Illuminate\Support\Facades\Storage::disk('public')->exists($certPath)) {
            \Illuminate\Support\Facades\Storage::disk('public')->delete($certPath);
            $logs[] = "Deleted via Storage facade: $certPath";
            $deleted = true;
        }
    } catch (\Exception $e) {
        $logs[] = "Storage error: " . $e->getMessage();
    }

    return $deleted;
}

// Process the deletion
if ($vesselCertificate->attachment_path) {
    // Handle both string and array formats
    if (is_string($vesselCertificate->attachment_path) && !is_array(json_decode($vesselCertificate->attachment_path, true))) {
        // Single attachment as string
        if ($attachmentIndex == 0) {
            $attachmentToDelete = $vesselCertificate->attachment_path;
            $logs[] = "Deleting single attachment: $attachmentToDelete";

            // Delete the file
            deleteFileAggressively($attachmentToDelete, $logs);

            // Update database
            $vesselCertificate->attachment_path = null;
            $vesselCertificate->save();

            // Return clean JSON response with logs
            header('Content-Type: application/json');
            echo json_encode([
                'success' => true,
                'message' => 'Attachment deleted successfully.',
                'logs' => $logs
            ]);
            exit;
        } else {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Invalid attachment index for single attachment',
                'logs' => $logs
            ]);
            exit;
        }
    } else {
        // Multiple attachments as array
        $attachments = is_array($vesselCertificate->attachment_path)
            ? $vesselCertificate->attachment_path
            : json_decode($vesselCertificate->attachment_path, true);

        if (is_array($attachments) && isset($attachments[$attachmentIndex])) {
            $attachmentToDelete = $attachments[$attachmentIndex];
            $logs[] = "Deleting attachment from array: $attachmentToDelete";

            // Delete the file
            deleteFileAggressively($attachmentToDelete, $logs);

            // Remove from array
            unset($attachments[$attachmentIndex]);

            // Reindex array
            $attachments = array_values($attachments);

            // Update database
            $vesselCertificate->attachment_path = !empty($attachments) ? json_encode($attachments) : null;
            $vesselCertificate->save();

            // Return clean JSON response with logs
            header('Content-Type: application/json');
            echo json_encode([
                'success' => true,
                'message' => 'Attachment deleted successfully.',
                'logs' => $logs
            ]);
            exit;
        } else {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Attachment not found in array',
                'logs' => $logs
            ]);
            exit;
        }
    }
} else {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => 'No attachments found',
        'logs' => $logs
    ]);
    exit;
}

} catch (\Exception $e) {
    // Log the error
    error_log('Error in delete_attachment.php: ' . $e->getMessage());

    // Return a JSON error response
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while deleting the attachment: ' . $e->getMessage(),
        'logs' => isset($logs) ? $logs : []
    ]);
    exit;
}