<?php
/**
 * Test page for direct attachment deletion
 */

// Set error reporting to show all errors
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database connection parameters
$host = 'localhost';
$dbname = 'vessel_certificates';
$username = 'root';
$password = '';

// Try to get database credentials from Laravel's .env file
if (file_exists(__DIR__ . '/../.env')) {
    $envFile = file_get_contents(__DIR__ . '/../.env');
    preg_match('/DB_HOST=([^\n]+)/', $envFile, $hostMatches);
    preg_match('/DB_DATABASE=([^\n]+)/', $envFile, $dbnameMatches);
    preg_match('/DB_USERNAME=([^\n]+)/', $envFile, $usernameMatches);
    preg_match('/DB_PASSWORD=([^\n]+)/', $envFile, $passwordMatches);
    
    if (!empty($hostMatches[1])) $host = trim($hostMatches[1]);
    if (!empty($dbnameMatches[1])) $dbname = trim($dbnameMatches[1]);
    if (!empty($usernameMatches[1])) $username = trim($usernameMatches[1]);
    if (!empty($passwordMatches[1])) $password = trim($passwordMatches[1]);
}

// Connect to the database
try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $connectionSuccess = true;
} catch (PDOException $e) {
    $connectionError = $e->getMessage();
    $connectionSuccess = false;
}

// Get all vessel certificates with attachments
$certificates = [];
if ($connectionSuccess) {
    try {
        $stmt = $pdo->query("
            SELECT vc.id, vc.attachment_path, vc.vessel_id, v.name as vessel_name, c.name as certificate_name
            FROM vessel_certificates vc
            JOIN vessels v ON vc.vessel_id = v.id
            JOIN certificates c ON vc.certificate_id = c.id
            WHERE vc.attachment_path IS NOT NULL
            ORDER BY v.name, c.name
        ");
        $certificates = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $queryError = $e->getMessage();
    }
}

// Handle deletion if requested
$deleteResult = null;
if (isset($_POST['delete']) && isset($_POST['cert_id']) && isset($_POST['attachment_index'])) {
    $certId = $_POST['cert_id'];
    $attachmentIndex = $_POST['attachment_index'];
    
    // Call our direct deletion script
    $postData = [
        'vessel_certificate_id' => $certId,
        'attachment_index' => $attachmentIndex
    ];
    
    $ch = curl_init('http://' . $_SERVER['HTTP_HOST'] . '/direct_delete_attachment.php');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    $deleteResult = [
        'http_code' => $httpCode,
        'response' => $response
    ];
}

// HTML output
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Direct Attachment Deletion Test</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-5">
        <h1>Direct Attachment Deletion Test</h1>
        
        <?php if (isset($connectionError)): ?>
            <div class="alert alert-danger">
                <strong>Database Connection Error:</strong> <?php echo htmlspecialchars($connectionError); ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($queryError)): ?>
            <div class="alert alert-danger">
                <strong>Query Error:</strong> <?php echo htmlspecialchars($queryError); ?>
            </div>
        <?php endif; ?>
        
        <?php if ($deleteResult): ?>
            <div class="alert <?php echo $deleteResult['http_code'] == 200 ? 'alert-success' : 'alert-danger'; ?>">
                <h4>Deletion Result (HTTP Code: <?php echo $deleteResult['http_code']; ?>)</h4>
                <pre><?php echo htmlspecialchars($deleteResult['response']); ?></pre>
            </div>
        <?php endif; ?>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2>Certificates with Attachments</h2>
            </div>
            <div class="card-body">
                <?php if (empty($certificates)): ?>
                    <p>No certificates with attachments found.</p>
                <?php else: ?>
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Vessel</th>
                                <th>Certificate</th>
                                <th>Attachment Path</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($certificates as $cert): ?>
                                <?php
                                $attachments = json_decode($cert['attachment_path'], true);
                                if (!is_array($attachments)) {
                                    // Single attachment
                                    $attachments = [$cert['attachment_path']];
                                }
                                ?>
                                <?php foreach ($attachments as $index => $attachment): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($cert['id']); ?></td>
                                        <td><?php echo htmlspecialchars($cert['vessel_name']); ?></td>
                                        <td><?php echo htmlspecialchars($cert['certificate_name']); ?></td>
                                        <td>
                                            <small><?php echo htmlspecialchars($attachment); ?></small>
                                        </td>
                                        <td>
                                            <form method="post" onsubmit="return confirm('Are you sure you want to delete this attachment?');">
                                                <input type="hidden" name="cert_id" value="<?php echo htmlspecialchars($cert['id']); ?>">
                                                <input type="hidden" name="attachment_index" value="<?php echo $index; ?>">
                                                <button type="submit" name="delete" class="btn btn-sm btn-danger">
                                                    Delete
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Database Connection Info</h2>
            </div>
            <div class="card-body">
                <p><strong>Host:</strong> <?php echo htmlspecialchars($host); ?></p>
                <p><strong>Database:</strong> <?php echo htmlspecialchars($dbname); ?></p>
                <p><strong>Username:</strong> <?php echo htmlspecialchars($username); ?></p>
                <p><strong>Connection Status:</strong> <?php echo $connectionSuccess ? '<span class="text-success">Connected</span>' : '<span class="text-danger">Failed</span>'; ?></p>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
