/**
 * Fix for Bootstrap modal accessibility issues
 * This script properly manages focus when modals are opened and closed
 * to prevent the "Blocked aria-hidden on an element because its descendant retained focus" error
 *
 * It uses the 'inert' attribute instead of 'aria-hidden' for better accessibility
 */
document.addEventListener('DOMContentLoaded', function() {
    // Store the element that had focus before the modal was opened
    let lastFocusedElement = null;

    // Find all modals and modify them to use inert instead of aria-hidden
    const modals = document.querySelectorAll('.modal');
    modals.forEach(modal => {
        // Add inert attribute to closed modals
        if (!modal.classList.contains('show')) {
            modal.inert = true;
        }

        // Remove aria-hidden attribute from all modals
        if (modal.hasAttribute('aria-hidden')) {
            modal.removeAttribute('aria-hidden');
        }

        // Create a MutationObserver to watch for aria-hidden being added
        const observer = new MutationObserver(mutations => {
            mutations.forEach(mutation => {
                if (mutation.type === 'attributes' && mutation.attributeName === 'aria-hidden') {
                    // If aria-hidden is added, remove it and use inert instead
                    if (modal.getAttribute('aria-hidden') === 'true') {
                        modal.removeAttribute('aria-hidden');
                        modal.inert = true;
                    } else if (modal.getAttribute('aria-hidden') === 'false') {
                        modal.removeAttribute('aria-hidden');
                        modal.inert = false;
                    }
                }
            });
        });

        // Start observing the modal for aria-hidden changes
        observer.observe(modal, { attributes: true });
    });

    // Handle modal open events
    document.addEventListener('show.bs.modal', function(event) {
        // Store the element that had focus before the modal was opened
        lastFocusedElement = document.activeElement;

        // Get the modal that is about to be opened
        const modal = event.target;

        // Remove inert attribute when modal is shown
        modal.inert = false;
    });

    // Handle modal shown events (after the modal is fully opened)
    document.addEventListener('shown.bs.modal', function(event) {
        // Get the modal that was just opened
        const modal = event.target;

        // Find the first focusable element in the modal and focus it
        const focusableElements = modal.querySelectorAll(
            'button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])'
        );

        if (focusableElements.length > 0) {
            focusableElements[0].focus();
        }
    });

    // Handle modal hide events (before the modal is fully closed)
    document.addEventListener('hide.bs.modal', function(event) {
        // Clear any focused elements inside the modal
        const modal = event.target;
        const focusedElementInModal = modal.querySelector(':focus');

        if (focusedElementInModal) {
            focusedElementInModal.blur();
        }
    });

    // Handle modal close events (after the modal is fully closed)
    document.addEventListener('hidden.bs.modal', function(event) {
        // Get the modal that was just closed
        const modal = event.target;

        // Add inert attribute when modal is hidden
        modal.inert = true;

        // Restore focus to the element that had focus before the modal was opened
        if (lastFocusedElement) {
            setTimeout(() => {
                lastFocusedElement.focus();
                lastFocusedElement = null;
            }, 0);
        }
    });
});
