@extends('layouts.app')

@section('title', 'Database Management')

@push('head')
<meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
<meta http-equiv="Pragma" content="no-cache">
<meta http-equiv="Expires" content="0">
@endpush

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1>Database Management</h1>
                <a href="{{ route('settings.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Settings
                </a>
            </div>

            @if(session('success'))
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    {{ session('success') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            @if(session('error'))
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    {{ session('error') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            <!-- Duplicate Statistics -->
            @if($duplicateStats->count() > 0)
                <div class="alert alert-warning">
                    <h5><i class="fas fa-exclamation-triangle"></i> Duplicate Certificates Found</h5>
                    <p>The following vessel-certificate combinations have duplicate records:</p>
                    <ul class="mb-0">
                        @foreach($duplicateStats as $duplicate)
                            <li>
                                <strong>{{ $duplicate->vessel->name }}</strong> - 
                                <strong>{{ $duplicate->certificate->name }}</strong> 
                                ({{ $duplicate->count }} records)
                            </li>
                        @endforeach
                    </ul>
                    <div class="mt-2">
                        <a href="{{ route('settings.database', ['show_duplicates' => 1]) }}" class="btn btn-sm btn-warning">
                            <i class="fas fa-eye"></i> View Duplicates Only
                        </a>
                    </div>
                </div>
            @endif

            <!-- Filters -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5><i class="fas fa-filter"></i> Filters</h5>
                </div>
                <div class="card-body">
                    <form method="GET" action="{{ route('settings.database') }}">
                        <div class="row">
                            <div class="col-md-3">
                                <label for="vessel_id" class="form-label">Vessel</label>
                                <select name="vessel_id" id="vessel_id" class="form-select">
                                    <option value="">All Vessels</option>
                                    @foreach($vessels as $vessel)
                                        <option value="{{ $vessel->id }}" {{ request('vessel_id') == $vessel->id ? 'selected' : '' }}>
                                            {{ $vessel->name }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="certificate_id" class="form-label">Certificate</label>
                                <select name="certificate_id" id="certificate_id" class="form-select">
                                    <option value="">All Certificates</option>
                                    @foreach($certificates as $certificate)
                                        <option value="{{ $certificate->id }}" {{ request('certificate_id') == $certificate->id ? 'selected' : '' }}>
                                            {{ $certificate->name }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label for="status" class="form-label">Status</label>
                                <select name="status" id="status" class="form-select">
                                    <option value="">All Status</option>
                                    <option value="expired" {{ request('status') == 'expired' ? 'selected' : '' }}>Expired</option>
                                    <option value="expiring_soon" {{ request('status') == 'expiring_soon' ? 'selected' : '' }}>Expiring Soon</option>
                                    <option value="valid" {{ request('status') == 'valid' ? 'selected' : '' }}>Valid</option>
                                    <option value="no_expiry" {{ request('status') == 'no_expiry' ? 'selected' : '' }}>No Expiry</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label for="search" class="form-label">Search</label>
                                <input type="text" name="search" id="search" class="form-control" 
                                       placeholder="ID, Vessel, Certificate..." value="{{ request('search') }}">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search"></i> Filter
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-md-12">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" name="show_duplicates" value="1" 
                                           id="show_duplicates" {{ request('show_duplicates') ? 'checked' : '' }}>
                                    <label class="form-check-label" for="show_duplicates">
                                        Show only duplicate records
                                    </label>
                                </div>
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-md-12">
                                <a href="{{ route('settings.database') }}" class="btn btn-outline-secondary btn-sm">
                                    <i class="fas fa-times"></i> Clear Filters
                                </a>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Results -->
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5><i class="fas fa-table"></i> Vessel Certificate Records</h5>
                    <span class="badge bg-info">{{ $vesselCertificates->count() }} records</span>
                </div>
                <div class="card-body">
                    @if($vesselCertificates->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Vessel</th>
                                        <th>Certificate</th>
                                        <th>ID Number</th>
                                        <th>Department</th>
                                        <th>Issued Date</th>
                                        <th>Expiry Date</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($vesselCertificates as $cert)
                                        <tr>
                                            <td>{{ $cert->id }}</td>
                                            <td>{{ $cert->vessel->name }}</td>
                                            <td>
                                                <span class="badge bg-secondary">{{ $cert->certificate->category }}</span>
                                                {{ $cert->certificate->name }}
                                            </td>
                                            <td>{{ $cert->id_no ?: '-' }}</td>
                                            <td>{{ $cert->vessel_department ?: '-' }}</td>
                                            <td>{{ $cert->issued_date ? $cert->issued_date->format('Y-m-d') : '-' }}</td>
                                            <td>{{ $cert->expiry_date ? $cert->expiry_date->format('Y-m-d') : '-' }}</td>
                                            <td>
                                                @if($cert->expiry_date)
                                                    @if($cert->expiry_date->isPast())
                                                        <span class="badge bg-danger">Expired</span>
                                                    @elseif($cert->expiry_date->lte(now()->addDays(30)))
                                                        <span class="badge bg-warning text-dark">Expiring Soon</span>
                                                    @else
                                                        <span class="badge bg-success">Valid</span>
                                                    @endif
                                                @else
                                                    <span class="badge bg-secondary">No Expiry</span>
                                                @endif
                                            </td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-danger delete-record-btn"
                                                        data-id="{{ $cert->id }}"
                                                        data-vessel="{{ $cert->vessel->name }}"
                                                        data-certificate="{{ $cert->certificate->name }}">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <div class="text-center py-4">
                            <i class="fas fa-search fa-3x text-muted mb-3"></i>
                            <h5>No records found</h5>
                            <p class="text-muted">Try adjusting your filters to see more results.</p>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this certificate record?</p>
                <div class="alert alert-warning">
                    <strong>Vessel:</strong> <span id="deleteVesselName"></span><br>
                    <strong>Certificate:</strong> <span id="deleteCertificateName"></span>
                </div>
                <p class="text-danger"><strong>This action cannot be undone!</strong></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteBtn">
                    <i class="fas fa-trash"></i> Delete Record
                </button>
            </div>
        </div>
    </div>
</div>

@section('scripts')
<script>
// Cache buster: {{ time() }}
// Set base URL for AJAX requests
const baseUrl = '{{ url('/') }}';
console.log('Base URL:', baseUrl);

document.addEventListener('DOMContentLoaded', function() {
    const deleteModal = new bootstrap.Modal(document.getElementById('deleteModal'));
    let currentRecordId = null;

    // Handle delete button clicks
    document.querySelectorAll('.delete-record-btn').forEach(button => {
        button.addEventListener('click', function() {
            currentRecordId = this.getAttribute('data-id');
            const vesselName = this.getAttribute('data-vessel');
            const certificateName = this.getAttribute('data-certificate');

            document.getElementById('deleteVesselName').textContent = vesselName;
            document.getElementById('deleteCertificateName').textContent = certificateName;

            deleteModal.show();
        });
    });

    // Handle confirm delete
    document.getElementById('confirmDeleteBtn').addEventListener('click', function() {
        if (!currentRecordId) return;

        const button = this;
        const originalText = button.innerHTML;

        // Show loading state
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Deleting...';
        button.disabled = true;

        // Send delete request
        const deleteUrl = `${baseUrl}/vessel-certificates/${currentRecordId}`;
        console.log('Deleting certificate with URL:', deleteUrl);

        fetch(deleteUrl, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        })
        .then(response => {
            console.log('Response status:', response.status);
            console.log('Response headers:', response.headers);

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            return response.text().then(text => {
                console.log('Raw response:', text);
                try {
                    return JSON.parse(text);
                } catch (e) {
                    console.error('Failed to parse JSON:', e);
                    throw new Error('Invalid JSON response: ' + text.substring(0, 100));
                }
            });
        })
        .then(data => {
            if (data.success) {
                // Close modal
                deleteModal.hide();

                // Show success message
                showAlert('success', data.message);

                // Remove the row from the table
                const row = document.querySelector(`tr:has(button[data-id="${currentRecordId}"])`);
                if (row) {
                    row.remove();
                }

                // Update record count if visible
                const badge = document.querySelector('.badge.bg-info');
                if (badge) {
                    const currentCount = parseInt(badge.textContent.split(' ')[0]);
                    badge.textContent = `${currentCount - 1} records`;
                }
            } else {
                showAlert('danger', data.message || 'Failed to delete record');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showAlert('danger', 'An error occurred while deleting the record');
        })
        .finally(() => {
            // Reset button state
            button.innerHTML = originalText;
            button.disabled = false;
            currentRecordId = null;
        });
    });

    // Function to show alerts
    function showAlert(type, message) {
        const alertHtml = `
            <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `;

        // Insert alert at the top of the container
        const container = document.querySelector('.container-fluid .row .col-12');
        const firstCard = container.querySelector('.card');
        firstCard.insertAdjacentHTML('beforebegin', alertHtml);

        // Auto-dismiss after 5 seconds
        setTimeout(() => {
            const alert = container.querySelector('.alert');
            if (alert) {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            }
        }, 5000);
    }
});
</script>
@endsection
