@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <div class="row mb-3">
        <div class="col-md-6">
            <h2>{{ $vessel->name }}</h2>
        </div>
        <div class="col-md-6 text-end">
            <a href="{{ route('vessels.index') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Back to Vessels
            </a>

            @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_certificates'))
            <a href="{{ route('vessels.edit', $vessel) }}" class="btn btn-warning">
                <i class="fas fa-edit"></i> Edit Vessel
            </a>
            @endif
        </div>
    </div>

    @if(session('success'))
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        {{ session('success') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    @endif

    @if(session('error'))
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        {{ session('error') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    @endif

    <div class="row">
        <div class="col-md-12">
            <ul class="nav nav-tabs" id="vesselTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="certificates-tab" data-bs-toggle="tab" data-bs-target="#certificates" type="button" role="tab" aria-controls="certificates" aria-selected="true">
                        <i class="fas fa-certificate"></i> Certificates
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="particulars-tab" data-bs-toggle="tab" data-bs-target="#particulars" type="button" role="tab" aria-controls="particulars" aria-selected="false">
                        <i class="fas fa-info-circle"></i> Particulars
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="plans-tab" data-bs-toggle="tab" data-bs-target="#plans" type="button" role="tab" aria-controls="plans" aria-selected="false">
                        <i class="fas fa-map"></i> Plans
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="surveys-tab" data-bs-toggle="tab" data-bs-target="#surveys" type="button" role="tab" aria-controls="surveys" aria-selected="false">
                        <i class="fas fa-clipboard-check"></i> Surveys
                    </button>
                </li>
            </ul>

            <div class="tab-content" id="vesselTabsContent">
                <!-- Certificates Tab -->
                <div class="tab-pane fade show active" id="certificates" role="tabpanel" aria-labelledby="certificates-tab">
                    <div class="card">
                        <div class="card-header">
                            <div class="row">
                                <div class="col-md-6">
                                    <h5>Certificates ({{ $filledCertificateCount }})</h5>
                                </div>
                                <div class="col-md-6 text-end">
                                    @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_certificates'))
                                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCertificateModal">
                                        <i class="fas fa-plus"></i> Add Certificate
                                    </button>
                                    @endif
                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <button class="btn btn-sm btn-success" id="exportToExcel">
                                    <i class="fas fa-file-excel"></i> Export to Excel
                                </button>
                                <button class="btn btn-sm btn-primary" id="toggleEmptyCertificates">
                                    <i class="fas fa-eye"></i> Show/Hide Empty Certificates
                                </button>

                                <!-- Bulk Download Controls -->
                                <div class="btn-group" role="group">
                                    <button type="button" class="btn btn-sm btn-info dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
                                        <i class="fas fa-download"></i> Bulk Download
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li><a class="dropdown-item" href="#" id="downloadSelected">Download Selected</a></li>
                                        <li><hr class="dropdown-divider"></li>
                                        <li><h6 class="dropdown-header">Download by Category</h6></li>
                                        @php
                                            $categories = $certificates->pluck('category')->unique()->sort();
                                        @endphp
                                        @foreach($categories as $category)
                                            <li><a class="dropdown-item download-category" href="#" data-category="{{ $category }}">{{ $category }}</a></li>
                                        @endforeach
                                    </ul>
                                </div>

                                <button class="btn btn-sm btn-secondary" id="selectAll">
                                    <i class="fas fa-check-square"></i> Select All
                                </button>
                                <button class="btn btn-sm btn-secondary" id="deselectAll">
                                    <i class="fas fa-square"></i> Deselect All
                                </button>

                                <span class="selected-count ms-3" id="selectedCount">0 selected</span>
                            </div>

                            <div class="table-responsive">
                                <table class="table table-striped table-hover" id="certificatesTable">
                                    <thead>
                                        <tr>
                                            <th>
                                                <input type="checkbox" id="selectAllCheckbox" class="form-check-input">
                                            </th>
                                            <th>Category</th>
                                            <th>Certificate</th>
                                            <th>ID No.</th>
                                            <th>Department</th>
                                            <th>Issued Date</th>
                                            <th>Expiry Date</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @php
                                            // Group certificates by category
                                            $certificatesByCategory = [];
                                            foreach ($certificates as $cert) {
                                                if (!isset($certificatesByCategory[$cert->category])) {
                                                    $certificatesByCategory[$cert->category] = [];
                                                }
                                                $certificatesByCategory[$cert->category][] = $cert;
                                            }

                                            // Sort categories by category_order
                                            ksort($certificatesByCategory);
                                        @endphp

                                        @foreach($certificatesByCategory as $category => $categoryCertificates)
                                            <tr class="category-header bg-dark text-white">
                                                <td colspan="9" class="py-2">
                                                    <h5 class="mb-0"><i class="fas fa-folder me-2"></i> CATEGORY: {{ $category }}</h5>
                                                </td>
                                            </tr>

                                            @foreach($categoryCertificates as $certificate)
                                                @php
                                                    $vesselCert = $vessel->vesselCertificates->where('certificate_id', $certificate->id)->first();
                                                    $isEmpty = !$vesselCert || (!$vesselCert->id_no && !$vesselCert->vessel_department && !$vesselCert->issued_date && !$vesselCert->expiry_date);
                                                @endphp

                                                <tr class="{{ $isEmpty ? 'empty-certificate' : '' }}" data-category="{{ $category }}">
                                                    <td>
                                                        @if($vesselCert && $vesselCert->attachment_path)
                                                            <input type="checkbox" class="form-check-input certificate-checkbox"
                                                                   data-cert-id="{{ $vesselCert->id }}"
                                                                   data-category="{{ $category }}"
                                                                   data-vessel-name="{{ $vessel->name }}"
                                                                   data-cert-name="{{ $certificate->name }}">
                                                        @endif
                                                    </td>
                                                    <td>{{ $category }}</td>
                                                    <td>{{ $certificate->name }}</td>
                                                    <td>{{ $vesselCert->id_no ?? '-' }}</td>
                                                    <td>{{ $vesselCert->vessel_department ?? '-' }}</td>
                                                    <td>{{ $vesselCert && $vesselCert->issued_date ? $vesselCert->issued_date->format('Y-m-d') : '-' }}</td>
                                                    <td>{{ $vesselCert && $vesselCert->expiry_date ? $vesselCert->expiry_date->format('Y-m-d') : '-' }}</td>
                                                    <td>
                                                        @if($vesselCert && $vesselCert->expiry_date)
                                                            @php
                                                                $warningSettings = \App\Models\WarningSettings::first();
                                                                $warningDays = $warningSettings ? $warningSettings->warning_days : 30;
                                                                $dangerDays = $warningSettings ? $warningSettings->danger_days : 7;
                                                                $daysUntilExpiry = now()->diffInDays($vesselCert->expiry_date, false);
                                                            @endphp
                                                            @if($vesselCert->expiry_date < now())
                                                                <span class="badge bg-danger" title="Expired {{ abs($daysUntilExpiry) }} days ago">Expired</span>
                                                            @elseif($vesselCert->expiry_date < now()->addDays($dangerDays))
                                                                <span class="badge bg-danger" title="Expires in {{ $daysUntilExpiry }} days (Danger: {{ $dangerDays }} days)">Expiring Soon</span>
                                                            @elseif($vesselCert->expiry_date < now()->addDays($warningDays))
                                                                <span class="badge bg-warning text-dark" title="Expires in {{ $daysUntilExpiry }} days (Warning: {{ $warningDays }} days)">Expiring Soon</span>
                                                            @else
                                                                <span class="badge bg-success" title="Expires in {{ $daysUntilExpiry }} days">Valid</span>
                                                            @endif
                                                        @else
                                                            <span class="badge bg-secondary">No Expiry</span>
                                                        @endif
                                                    </td>
                                                    <td>
                                                        <div class="btn-group" role="group">
                                                            @if($vesselCert)
                                                                <button type="button" class="btn btn-sm btn-info" data-bs-toggle="modal" data-bs-target="#viewCertificateModal{{ $vesselCert->id }}">
                                                                    <i class="fas fa-eye"></i> View
                                                                </button>

                                                                @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_certificates'))
                                                                <button type="button" class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#editCertificateModal{{ $vesselCert->id }}">
                                                                    <i class="fas fa-edit"></i> Edit
                                                                </button>
                                                                @endif
                                                            @else
                                                                @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_certificates'))
                                                                <button type="button" class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addCertificateModal"
                                                                        onclick="document.getElementById('certificate_id').value = '{{ $certificate->id }}'">
                                                                    <i class="fas fa-plus"></i> Add
                                                                </button>
                                                                @endif
                                                            @endif
                                                        </div>
                                                    </td>
                                                </tr>
                                            @endforeach
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>
                        </div>

                        <script>
                            document.addEventListener('DOMContentLoaded', function() {
                                // Toggle empty certificates
                                const toggleBtn = document.getElementById('toggleEmptyCertificates');
                                toggleBtn.addEventListener('click', function() {
                                    const emptyCerts = document.querySelectorAll('.empty-certificate');
                                    emptyCerts.forEach(cert => {
                                        cert.style.display = cert.style.display === 'none' ? '' : 'none';
                                    });
                                });

                                // Export to Excel
                                document.getElementById('exportToExcel').addEventListener('click', function() {
                                    // Create a workbook
                                    const wb = XLSX.utils.book_new();

                                    // Get the table
                                    const table = document.getElementById('certificatesTable');

                                    // Convert table to worksheet
                                    const ws = XLSX.utils.table_to_sheet(table);

                                    // Add worksheet to workbook
                                    XLSX.utils.book_append_sheet(wb, ws, 'Certificates');

                                    // Save workbook
                                    XLSX.writeFile(wb, '{{ $vessel->name }}_Certificates.xlsx');
                                });
                            });
                        </script>
                    </div>
                </div>

                <!-- Particulars Tab -->
                <div class="tab-pane fade" id="particulars" role="tabpanel" aria-labelledby="particulars-tab">
                    <div class="card">
                        <div class="card-header">
                            <h5>Vessel Particulars</h5>
                        </div>
                        <div class="card-body">
                            @if($vessel->particulars)
                                <div class="row mb-3">
                                    <div class="col-md-3">
                                        <strong>IMO Number:</strong>
                                    </div>
                                    <div class="col-md-9">
                                        {{ $vessel->particulars->imo_number ?? 'Not specified' }}
                                    </div>
                                </div>

                                @if($vessel->particulars->attachment_path)
                                <div class="row mb-3">
                                    <div class="col-md-3">
                                        <strong>Attachment:</strong>
                                    </div>
                                    <div class="col-md-9">
                                        <a href="{{ asset('storage/' . $vessel->particulars->attachment_path) }}" target="_blank" class="btn btn-sm btn-primary">
                                            <i class="fas fa-download"></i> Download Particulars
                                        </a>
                                    </div>
                                </div>
                                @endif

                                <div class="row mb-3">
                                    <div class="col-md-12">
                                        @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_certificates'))
                                        <button type="button" class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#editParticularsModal">
                                            <i class="fas fa-edit"></i> Edit Particulars
                                        </button>
                                        @endif
                                    </div>
                                </div>
                            @else
                                <p>No particulars information available for this vessel.</p>

                                @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_certificates'))
                                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addParticularsModal">
                                    <i class="fas fa-plus"></i> Add Particulars
                                </button>
                                @endif
                            @endif
                        </div>
                    </div>
                </div>

                <!-- Plans Tab -->
                <div class="tab-pane fade" id="plans" role="tabpanel" aria-labelledby="plans-tab">
                    <div class="card">
                        <div class="card-header">
                            <div class="row">
                                <div class="col-md-6">
                                    <h5>Vessel Plans</h5>
                                </div>
                                <div class="col-md-6 text-end">
                                    @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_plans'))
                                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addPlanModal">
                                        <i class="fas fa-plus"></i> Add Plan
                                    </button>
                                    @endif
                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <th>Plan</th>
                                            <th>Category</th>
                                            <th>Attachment</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @forelse($vessel->vesselPlans as $vesselPlan)
                                        <tr>
                                            <td>{{ $vesselPlan->plan->name }}</td>
                                            <td>{{ $vesselPlan->plan->category }}</td>
                                            <td>
                                                @if($vesselPlan->attachment_path)
                                                <a href="{{ asset('storage/' . $vesselPlan->attachment_path) }}" target="_blank" class="btn btn-sm btn-primary">
                                                    <i class="fas fa-download"></i> Download
                                                </a>
                                                @else
                                                <span class="text-muted">No attachment</span>
                                                @endif
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_plans'))
                                                    <button type="button" class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#editPlanModal{{ $vesselPlan->id }}">
                                                        <i class="fas fa-edit"></i> Edit
                                                    </button>

                                                    <button type="button" class="btn btn-sm btn-danger"
                                                            onclick="if(confirm('Are you sure you want to delete this plan?')) {
                                                                document.getElementById('delete-plan-{{ $vesselPlan->id }}').submit();
                                                            }">
                                                        <i class="fas fa-trash"></i> Delete
                                                    </button>

                                                    <form id="delete-plan-{{ $vesselPlan->id }}" action="{{ route('vessel-plans.destroy', $vesselPlan) }}" method="POST" style="display: none;">
                                                        @csrf
                                                        @method('DELETE')
                                                    </form>
                                                    @endif
                                                </div>
                                            </td>
                                        </tr>
                                        @empty
                                        <tr>
                                            <td colspan="4" class="text-center">No plans found for this vessel.</td>
                                        </tr>
                                        @endforelse
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Surveys Tab -->
                <div class="tab-pane fade" id="surveys" role="tabpanel" aria-labelledby="surveys-tab">
                    <div class="card">
                        <div class="card-header">
                            <div class="row">
                                <div class="col-md-6">
                                    <h5>Vessel Surveys</h5>
                                </div>
                                <div class="col-md-6 text-end">
                                    @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_surveys'))
                                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addSurveyModal">
                                        <i class="fas fa-plus"></i> Add Survey
                                    </button>
                                    @endif
                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <th>Survey</th>
                                            <th>Category</th>
                                            <th>Last Date</th>
                                            <th>Due Date</th>
                                            <th>Window Date</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @forelse($vessel->vesselSurveys as $vesselSurvey)
                                        <tr>
                                            <td>{{ $vesselSurvey->survey->name }}</td>
                                            <td>{{ $vesselSurvey->survey->category }}</td>
                                            <td>{{ $vesselSurvey->last_date ? $vesselSurvey->last_date->format('Y-m-d') : '-' }}</td>
                                            <td>{{ $vesselSurvey->due_date ? $vesselSurvey->due_date->format('Y-m-d') : '-' }}</td>
                                            <td>{{ $vesselSurvey->window_date ? $vesselSurvey->window_date->format('Y-m-d') : '-' }}</td>
                                            <td>
                                                @if($vesselSurvey->due_date)
                                                    @if($vesselSurvey->due_date < now())
                                                        <span class="badge bg-danger">Overdue</span>
                                                    @elseif($vesselSurvey->due_date < now()->addDays(90))
                                                        <span class="badge bg-warning text-dark">Due Soon</span>
                                                    @else
                                                        <span class="badge bg-success">Scheduled</span>
                                                    @endif
                                                @else
                                                    <span class="badge bg-secondary">No Due Date</span>
                                                @endif
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    <button type="button" class="btn btn-sm btn-info" data-bs-toggle="modal" data-bs-target="#viewSurveyModal{{ $vesselSurvey->id }}">
                                                        <i class="fas fa-eye"></i> View
                                                    </button>

                                                    @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_surveys'))
                                                    <button type="button" class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#editSurveyModal{{ $vesselSurvey->id }}">
                                                        <i class="fas fa-edit"></i> Edit
                                                    </button>

                                                    <button type="button" class="btn btn-sm btn-danger"
                                                            onclick="if(confirm('Are you sure you want to delete this survey?')) {
                                                                document.getElementById('delete-survey-{{ $vesselSurvey->id }}').submit();
                                                            }">
                                                        <i class="fas fa-trash"></i> Delete
                                                    </button>

                                                    <form id="delete-survey-{{ $vesselSurvey->id }}" action="{{ route('vessel-surveys.destroy', $vesselSurvey) }}" method="POST" style="display: none;">
                                                        @csrf
                                                        @method('DELETE')
                                                    </form>
                                                    @endif
                                                </div>
                                            </td>
                                        </tr>
                                        @empty
                                        <tr>
                                            <td colspan="7" class="text-center">No surveys found for this vessel.</td>
                                        </tr>
                                        @endforelse
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<!-- Modals -->

<!-- Add Certificate Modal -->
<div class="modal fade" id="addCertificateModal" tabindex="-1" aria-labelledby="addCertificateModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addCertificateModalLabel">Add Certificate</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-certificates.store') }}" method="POST" enctype="multipart/form-data">
                @csrf
                <div class="modal-body">
                    <input type="hidden" name="vessel_id" value="{{ $vessel->id }}">

                    <div class="mb-3">
                        <label for="certificate_id" class="form-label">Certificate</label>
                        <select class="form-select" id="certificate_id" name="certificate_id" required>
                            <option value="">Select Certificate</option>
                            @foreach($certificates as $certificate)
                            <option value="{{ $certificate->id }}">{{ $certificate->name }} ({{ $certificate->category }})</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="id_no" class="form-label">ID Number</label>
                        <input type="text" class="form-control" id="id_no" name="id_no">
                    </div>

                    <div class="mb-3">
                        <label for="vessel_department" class="form-label">Department</label>
                        <input type="text" class="form-control" id="vessel_department" name="vessel_department">
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="issued_date" class="form-label">Issued Date</label>
                                <input type="date" class="form-control" id="issued_date" name="issued_date">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="expiry_date" class="form-label">Expiry Date</label>
                                <input type="date" class="form-control" id="expiry_date" name="expiry_date">
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="attachment" class="form-label">Attachments</label>
                        <input type="file" class="form-control" id="attachment" name="attachments[]" multiple>
                        <div class="form-text">You can select multiple files at once</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Certificate</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Particulars Modal -->
<div class="modal fade" id="addParticularsModal" tabindex="-1" aria-labelledby="addParticularsModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addParticularsModalLabel">Add Vessel Particulars</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-particulars.store') }}" method="POST" enctype="multipart/form-data">
                @csrf
                <div class="modal-body">
                    <input type="hidden" name="vessel_id" value="{{ $vessel->id }}">

                    <div class="mb-3">
                        <label for="imo_number" class="form-label">IMO Number</label>
                        <input type="text" class="form-control" id="imo_number" name="imo_number">
                    </div>

                    <div class="mb-3">
                        <label for="attachment" class="form-label">Attachment</label>
                        <input type="file" class="form-control" id="attachment" name="attachment">
                        <div class="form-text">Upload vessel particulars document (PDF, DOC, etc.)</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Particulars</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Particulars Modal -->
@if($vessel->particulars)
<div class="modal fade" id="editParticularsModal" tabindex="-1" aria-labelledby="editParticularsModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editParticularsModalLabel">Edit Vessel Particulars</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-particulars.update', $vessel->particulars) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit_imo_number" class="form-label">IMO Number</label>
                        <input type="text" class="form-control" id="edit_imo_number" name="imo_number" value="{{ $vessel->particulars->imo_number }}">
                    </div>

                    <div class="mb-3">
                        <label for="edit_attachment" class="form-label">Attachment</label>
                        <input type="file" class="form-control" id="edit_attachment" name="attachment">
                        <div class="form-text">Upload new vessel particulars document (PDF, DOC, etc.)</div>

                        @if($vessel->particulars->attachment_path)
                        <div class="mt-2">
                            <strong>Current attachment:</strong>
                            <a href="{{ asset('storage/' . $vessel->particulars->attachment_path) }}" target="_blank">
                                View current attachment
                            </a>
                        </div>
                        @endif
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Particulars</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endif

<!-- Add Plan Modal -->
<div class="modal fade" id="addPlanModal" tabindex="-1" aria-labelledby="addPlanModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addPlanModalLabel">Add Plan</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-plans.store') }}" method="POST" enctype="multipart/form-data">
                @csrf
                <div class="modal-body">
                    <input type="hidden" name="vessel_id" value="{{ $vessel->id }}">

                    <div class="mb-3">
                        <label for="plan_id" class="form-label">Plan</label>
                        <select class="form-select" id="plan_id" name="plan_id" required>
                            <option value="">Select Plan</option>
                            @foreach($plans as $plan)
                            <option value="{{ $plan->id }}">{{ $plan->name }} ({{ $plan->category }})</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="attachment" class="form-label">Attachment</label>
                        <input type="file" class="form-control" id="attachment" name="attachment">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Plan</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Survey Modal -->
<div class="modal fade" id="addSurveyModal" tabindex="-1" aria-labelledby="addSurveyModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addSurveyModalLabel">Add Survey</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-surveys.store') }}" method="POST">
                @csrf
                <div class="modal-body">
                    <input type="hidden" name="vessel_id" value="{{ $vessel->id }}">

                    <div class="mb-3">
                        <label for="survey_id" class="form-label">Survey</label>
                        <select class="form-select" id="survey_id" name="survey_id" required>
                            <option value="">Select Survey</option>
                            @foreach($surveys as $survey)
                            <option value="{{ $survey->id }}">{{ $survey->name }} ({{ $survey->category }})</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="last_date" class="form-label">Last Date</label>
                                <input type="date" class="form-control" id="last_date" name="last_date">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="due_date" class="form-label">Due Date</label>
                                <input type="date" class="form-control" id="due_date" name="due_date">
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="window_date" class="form-label">Window Date</label>
                        <input type="date" class="form-control" id="window_date" name="window_date">
                    </div>

                    <div class="mb-3">
                        <label for="status" class="form-label">Status</label>
                        <input type="text" class="form-control" id="status" name="status">
                    </div>

                    <div class="mb-3">
                        <label for="general_note" class="form-label">General Note</label>
                        <textarea class="form-control" id="general_note" name="general_note" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Survey</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Plan Modals -->
@foreach($vessel->vesselPlans as $vesselPlan)
<div class="modal fade" id="editPlanModal{{ $vesselPlan->id }}" tabindex="-1" aria-labelledby="editPlanModalLabel{{ $vesselPlan->id }}" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editPlanModalLabel{{ $vesselPlan->id }}">Edit Plan</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-plans.update', $vesselPlan) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="plan_id{{ $vesselPlan->id }}" class="form-label">Plan</label>
                        <select class="form-select" id="plan_id{{ $vesselPlan->id }}" name="plan_id" required>
                            @foreach($plans as $plan)
                            <option value="{{ $plan->id }}" {{ $vesselPlan->plan_id == $plan->id ? 'selected' : '' }}>
                                {{ $plan->name }} ({{ $plan->category }})
                            </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="attachment{{ $vesselPlan->id }}" class="form-label">Attachment</label>
                        <input type="file" class="form-control" id="attachment{{ $vesselPlan->id }}" name="attachment">
                        <div class="form-text">Upload new plan document (PDF, DOC, etc.)</div>

                        @if($vesselPlan->attachment_path)
                        <div class="mt-2">
                            <strong>Current attachment:</strong>
                            <a href="{{ asset('storage/' . $vesselPlan->attachment_path) }}" target="_blank">
                                View current attachment
                            </a>
                        </div>
                        @endif
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Plan</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endforeach

<!-- Edit Survey Modals -->
@foreach($vessel->vesselSurveys as $vesselSurvey)
<div class="modal fade" id="editSurveyModal{{ $vesselSurvey->id }}" tabindex="-1" aria-labelledby="editSurveyModalLabel{{ $vesselSurvey->id }}" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editSurveyModalLabel{{ $vesselSurvey->id }}">Edit Survey</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-surveys.update', $vesselSurvey) }}" method="POST">
                @csrf
                @method('PUT')
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="survey_id{{ $vesselSurvey->id }}" class="form-label">Survey</label>
                        <select class="form-select" id="survey_id{{ $vesselSurvey->id }}" name="survey_id" required>
                            @foreach($surveys as $survey)
                            <option value="{{ $survey->id }}" {{ $vesselSurvey->survey_id == $survey->id ? 'selected' : '' }}>
                                {{ $survey->name }} ({{ $survey->category }})
                            </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="last_date{{ $vesselSurvey->id }}" class="form-label">Last Date</label>
                                <input type="date" class="form-control" id="last_date{{ $vesselSurvey->id }}" name="last_date" value="{{ $vesselSurvey->last_date ? $vesselSurvey->last_date->format('Y-m-d') : '' }}">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="due_date{{ $vesselSurvey->id }}" class="form-label">Due Date</label>
                                <input type="date" class="form-control" id="due_date{{ $vesselSurvey->id }}" name="due_date" value="{{ $vesselSurvey->due_date ? $vesselSurvey->due_date->format('Y-m-d') : '' }}">
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="window_date{{ $vesselSurvey->id }}" class="form-label">Window Date</label>
                        <input type="date" class="form-control" id="window_date{{ $vesselSurvey->id }}" name="window_date" value="{{ $vesselSurvey->window_date ? $vesselSurvey->window_date->format('Y-m-d') : '' }}">
                    </div>

                    <div class="mb-3">
                        <label for="status{{ $vesselSurvey->id }}" class="form-label">Status</label>
                        <input type="text" class="form-control" id="status{{ $vesselSurvey->id }}" name="status" value="{{ $vesselSurvey->status }}">
                    </div>

                    <div class="mb-3">
                        <label for="general_note{{ $vesselSurvey->id }}" class="form-label">General Note</label>
                        <textarea class="form-control" id="general_note{{ $vesselSurvey->id }}" name="general_note" rows="3">{{ $vesselSurvey->general_note }}</textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Survey</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- View Survey Modals -->
<div class="modal fade" id="viewSurveyModal{{ $vesselSurvey->id }}" tabindex="-1" aria-labelledby="viewSurveyModalLabel{{ $vesselSurvey->id }}" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewSurveyModalLabel{{ $vesselSurvey->id }}">
                    {{ $vesselSurvey->survey->name }}
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row mb-3">
                    <div class="col-md-4">
                        <strong>Survey:</strong>
                    </div>
                    <div class="col-md-8">
                        {{ $vesselSurvey->survey->name }} ({{ $vesselSurvey->survey->category }})
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-4">
                        <strong>Last Date:</strong>
                    </div>
                    <div class="col-md-8">
                        {{ $vesselSurvey->last_date ? $vesselSurvey->last_date->format('Y-m-d') : 'Not specified' }}
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-4">
                        <strong>Due Date:</strong>
                    </div>
                    <div class="col-md-8">
                        {{ $vesselSurvey->due_date ? $vesselSurvey->due_date->format('Y-m-d') : 'Not specified' }}
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-4">
                        <strong>Window Date:</strong>
                    </div>
                    <div class="col-md-8">
                        {{ $vesselSurvey->window_date ? $vesselSurvey->window_date->format('Y-m-d') : 'Not specified' }}
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-4">
                        <strong>Status:</strong>
                    </div>
                    <div class="col-md-8">
                        @if($vesselSurvey->status)
                            {{ $vesselSurvey->status }}
                        @else
                            @if($vesselSurvey->due_date)
                                @if($vesselSurvey->due_date < now())
                                    <span class="badge bg-danger">Overdue</span>
                                @elseif($vesselSurvey->due_date < now()->addDays(90))
                                    <span class="badge bg-warning text-dark">Due Soon</span>
                                @else
                                    <span class="badge bg-success">Scheduled</span>
                                @endif
                            @else
                                <span class="badge bg-secondary">No Due Date</span>
                            @endif
                        @endif
                    </div>
                </div>

                @if($vesselSurvey->general_note)
                <div class="row mb-3">
                    <div class="col-md-4">
                        <strong>General Note:</strong>
                    </div>
                    <div class="col-md-8">
                        {{ $vesselSurvey->general_note }}
                    </div>
                </div>
                @endif
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>
@endforeach

<!-- View Certificate Modals -->
@foreach($vessel->vesselCertificates as $vesselCert)
<div class="modal fade" id="viewCertificateModal{{ $vesselCert->id }}" tabindex="-1" aria-labelledby="viewCertificateModalLabel{{ $vesselCert->id }}" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewCertificateModalLabel{{ $vesselCert->id }}">
                    @if($vesselCert->certificate)
                        {{ $vesselCert->certificate->name }}
                    @else
                        Certificate Details
                    @endif
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row mb-3">
                    <div class="col-md-3">
                        <strong>Certificate:</strong>
                    </div>
                    <div class="col-md-9">
                        @if($vesselCert->certificate)
                            {{ $vesselCert->certificate->name }} ({{ $vesselCert->certificate->category }})
                        @else
                            <span class="text-danger">Missing Certificate Type</span>
                        @endif
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-3">
                        <strong>ID Number:</strong>
                    </div>
                    <div class="col-md-9">
                        {{ $vesselCert->id_no ?: 'Not specified' }}
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-3">
                        <strong>Department:</strong>
                    </div>
                    <div class="col-md-9">
                        {{ $vesselCert->vessel_department ?: 'Not specified' }}
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-3">
                        <strong>Issued Date:</strong>
                    </div>
                    <div class="col-md-9">
                        {{ $vesselCert->issued_date ? $vesselCert->issued_date->format('Y-m-d') : 'Not specified' }}
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-3">
                        <strong>Expiry Date:</strong>
                    </div>
                    <div class="col-md-9">
                        {{ $vesselCert->expiry_date ? $vesselCert->expiry_date->format('Y-m-d') : 'Not specified' }}
                    </div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-3">
                        <strong>Status:</strong>
                    </div>
                    <div class="col-md-9">
                        @if($vesselCert->expiry_date)
                            @if($vesselCert->expiry_date < now())
                                <span class="badge bg-danger">Expired</span>
                            @elseif($vesselCert->expiry_date < now()->addDays(30))
                                <span class="badge bg-warning text-dark">Expiring Soon</span>
                            @else
                                <span class="badge bg-success">Valid</span>
                            @endif
                        @else
                            <span class="badge bg-secondary">No Expiry</span>
                        @endif
                    </div>
                </div>

                @if($vesselCert->attachment_path)
                <div class="row mb-3">
                    <div class="col-md-3">
                        <strong>Attachments:</strong>
                    </div>
                    <div class="col-md-9">
                        @php
                            $attachments = json_decode($vesselCert->attachment_path, true);
                        @endphp

                        @if(is_array($attachments))
                            @foreach($attachments as $index => $attachment)
                                <div class="mb-2">
                                    @php
                                        $attachmentPath = $attachment;
                                    @endphp
                                    <div class="btn-group">
                                        <a href="{{ route('certificate.preview', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-info">
                                            <i class="fas fa-eye"></i> View
                                        </a>
                                        <a href="{{ route('certificate.file', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-primary">
                                            <i class="fas fa-download"></i> Download
                                        </a>
                                    </div>
                                </div>
                            @endforeach
                        @else
                            @php
                                $attachmentPath = $vesselCert->attachment_path;
                            @endphp
                            <div class="btn-group">
                                <a href="{{ route('certificate.preview', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-info">
                                    <i class="fas fa-eye"></i> View
                                </a>
                                <a href="{{ route('certificate.file', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-primary">
                                    <i class="fas fa-download"></i> Download
                                </a>
                            </div>
                        @endif
                    </div>
                </div>
                @endif
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>

                @if(auth()->user()->is_admin || auth()->user()->hasPermission('manage_certificates'))
                <button type="button" class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#editCertificateModal{{ $vesselCert->id }}" data-bs-dismiss="modal">
                    <i class="fas fa-edit"></i> Edit
                </button>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Edit Certificate Modal -->
<div class="modal fade" id="editCertificateModal{{ $vesselCert->id }}" tabindex="-1" aria-labelledby="editCertificateModalLabel{{ $vesselCert->id }}" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editCertificateModalLabel{{ $vesselCert->id }}">
                    Edit Certificate:
                    @if($vesselCert->certificate)
                        {{ $vesselCert->certificate->name }}
                    @else
                        Unknown Certificate
                    @endif
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('vessel-certificates.update', $vesselCert) }}" method="POST" enctype="multipart/form-data" id="update-certificate-form-{{ $vesselCert->id }}">
                @csrf
                @method('PUT')
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="id_no{{ $vesselCert->id }}" class="form-label">ID Number</label>
                        <input type="text" class="form-control" id="id_no{{ $vesselCert->id }}" name="id_no" value="{{ $vesselCert->id_no }}">
                    </div>

                    <div class="mb-3">
                        <label for="vessel_department{{ $vesselCert->id }}" class="form-label">Department</label>
                        <input type="text" class="form-control" id="vessel_department{{ $vesselCert->id }}" name="vessel_department" value="{{ $vesselCert->vessel_department }}">
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="issued_date{{ $vesselCert->id }}" class="form-label">Issued Date</label>
                                <input type="date" class="form-control" id="issued_date{{ $vesselCert->id }}" name="issued_date" value="{{ $vesselCert->issued_date ? $vesselCert->issued_date->format('Y-m-d') : '' }}">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="expiry_date{{ $vesselCert->id }}" class="form-label">Expiry Date</label>
                                <input type="date" class="form-control" id="expiry_date{{ $vesselCert->id }}" name="expiry_date" value="{{ $vesselCert->expiry_date ? $vesselCert->expiry_date->format('Y-m-d') : '' }}">
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="attachment{{ $vesselCert->id }}" class="form-label">Add Attachments</label>
                        <input type="file" class="form-control" id="attachment{{ $vesselCert->id }}" name="attachments[]" multiple>
                        <div class="form-text">You can select multiple files at once</div>
                    </div>

                    @if($vesselCert->attachment_path)
                    <div class="mb-3">
                        <label class="form-label">Current Attachments:</label>
                        @php
                            $attachments = json_decode($vesselCert->attachment_path, true);
                        @endphp

                        @if(is_array($attachments))
                            @foreach($attachments as $index => $attachment)
                                <div class="mb-2 d-flex align-items-center justify-content-between">
                                    @php
                                        $attachmentPath = $attachment;
                                        $fileName = basename($attachmentPath);
                                    @endphp
                                    <span class="text-muted small">{{ $fileName }}</span>
                                    <div class="btn-group">
                                        <a href="{{ route('certificate.preview', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-info">
                                            <i class="fas fa-eye"></i> View
                                        </a>
                                        <a href="{{ route('certificate.file', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-primary">
                                            <i class="fas fa-download"></i> Download
                                        </a>
                                        <form action="{{ url('/direct_delete_attachment.php') }}" method="POST" class="d-inline delete-attachment-form" onsubmit="return false;">
                                            @csrf
                                            <input type="hidden" name="vessel_certificate_id" value="{{ $vesselCert->id }}">
                                            <input type="hidden" name="attachment_index" value="{{ $index }}">
                                            <button type="button" class="btn btn-sm btn-danger simple-delete-btn"
                                                    data-cert-id="{{ $vesselCert->id }}"
                                                    data-index="{{ $index }}"
                                                    data-attachment="{{ $attachment }}">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            @endforeach
                        @else
                            @php
                                $attachmentPath = $vesselCert->attachment_path;
                                $fileName = basename($attachmentPath);
                            @endphp
                            <div class="d-flex align-items-center justify-content-between">
                                <span class="text-muted small">{{ $fileName }}</span>
                                <div class="btn-group">
                                    <a href="{{ route('certificate.preview', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-info">
                                        <i class="fas fa-eye"></i> View
                                    </a>
                                    <a href="{{ route('certificate.file', ['path' => $attachmentPath]) }}" target="_blank" class="btn btn-sm btn-primary">
                                        <i class="fas fa-download"></i> Download
                                    </a>
                                    <form action="{{ url('/direct_delete_attachment.php') }}" method="POST" class="d-inline delete-attachment-form" onsubmit="return false;">
                                        @csrf
                                        <input type="hidden" name="vessel_certificate_id" value="{{ $vesselCert->id }}">
                                        <input type="hidden" name="attachment_index" value="0">
                                        <button type="button" class="btn btn-sm btn-danger simple-delete-btn"
                                                data-cert-id="{{ $vesselCert->id }}"
                                                data-index="0"
                                                data-attachment="{{ $attachmentPath }}">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                    </form>
                                </div>
                            </div>
                        @endif
                    </div>
                    @endif
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary update-certificate-btn" data-id="{{ $vesselCert->id }}">Update Certificate</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endforeach

@endsection

@section('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Handle certificate update button clicks
    const updateCertificateBtns = document.querySelectorAll('.update-certificate-btn');

    updateCertificateBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const certId = btn.getAttribute('data-id');
            const form = document.getElementById('update-certificate-form-' + certId);

            if (!form) {
                console.error('Form not found for certificate ID: ' + certId);
                return;
            }

            // Show loading state
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
            btn.disabled = true;

            // Create FormData from the form
            const formData = new FormData(form);

            // Check if there are files to upload
            const fileInput = form.querySelector('input[type="file"]');
            const hasFiles = fileInput && fileInput.files.length > 0;

            // If there are files, check their size
            if (hasFiles) {
                let totalSize = 0;
                for (let i = 0; i < fileInput.files.length; i++) {
                    totalSize += fileInput.files[i].size;
                }

                // If total size is too large, show warning and remove files from form data
                if (totalSize > 32 * 1024 * 1024) { // 32MB limit
                    if (!confirm('The total file size exceeds 32MB, which may cause errors. Do you want to continue without uploading files?')) {
                        btn.innerHTML = originalText;
                        btn.disabled = false;
                        return;
                    }

                    // Remove files from form data
                    formData.delete('attachments[]');
                }
            }

            // Submit the form using fetch API
            fetch(form.action, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                if (data && data.success) {
                    // Show success message without alert to avoid modal closing
                    const successMessage = document.createElement('div');
                    successMessage.className = 'alert alert-success mt-2';
                    successMessage.textContent = 'Certificate updated successfully!';
                    successMessage.id = 'certificate-update-success-' + certId;

                    // Find a good place to insert the message
                    const modalBody = form.querySelector('.modal-body');
                    if (modalBody) {
                        // Remove any existing messages
                        const existingMessages = modalBody.querySelectorAll('.alert');
                        existingMessages.forEach(msg => msg.remove());

                        // Insert at the top of the modal body
                        modalBody.insertBefore(successMessage, modalBody.firstChild);

                        // After 2 seconds, close the modal and reload the page
                        setTimeout(() => {
                            // Close the modal
                            const modal = form.closest('.modal');
                            const modalInstance = bootstrap.Modal.getInstance(modal);
                            if (modalInstance) {
                                modalInstance.hide();
                            }

                            // Reload the page
                            window.location.reload();
                        }, 2000);
                    } else {
                        // Fallback if modal body not found
                        alert('Certificate updated successfully!');

                        // Close the modal
                        const modal = form.closest('.modal');
                        const modalInstance = bootstrap.Modal.getInstance(modal);
                        if (modalInstance) {
                            modalInstance.hide();
                        }

                        // Reload the page
                        window.location.reload();
                    }
                } else if (data && data.error) {
                    // Show error message without alert
                    const errorMessage = document.createElement('div');
                    errorMessage.className = 'alert alert-danger mt-2';
                    errorMessage.textContent = 'Error: ' + data.error;
                    errorMessage.id = 'certificate-update-error-' + certId;

                    const modalBody = form.querySelector('.modal-body');
                    if (modalBody) {
                        // Remove any existing messages
                        const existingMessages = modalBody.querySelectorAll('.alert');
                        existingMessages.forEach(msg => msg.remove());

                        modalBody.insertBefore(errorMessage, modalBody.firstChild);
                    } else {
                        // Fallback if modal body not found
                        alert('Error: ' + data.error);
                    }
                } else {
                    // Show generic error message without alert
                    const errorMessage = document.createElement('div');
                    errorMessage.className = 'alert alert-danger mt-2';
                    errorMessage.textContent = 'Unknown error occurred. Please try again.';
                    errorMessage.id = 'certificate-update-error-generic-' + certId;

                    const modalBody = form.querySelector('.modal-body');
                    if (modalBody) {
                        // Remove any existing messages
                        const existingMessages = modalBody.querySelectorAll('.alert');
                        existingMessages.forEach(msg => msg.remove());

                        modalBody.insertBefore(errorMessage, modalBody.firstChild);
                    } else {
                        // Fallback if modal body not found
                        alert('Unknown error occurred. Please try again.');
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);

                // Show error message without alert to avoid modal closing
                const errorMessage = document.createElement('div');
                errorMessage.className = 'alert alert-danger mt-2';
                errorMessage.textContent = 'An error occurred while updating the certificate. This may be because the file is too large. Please try again with smaller files (less than 32MB total).';
                errorMessage.id = 'certificate-update-error-network-' + certId;

                const modalBody = form.querySelector('.modal-body');
                if (modalBody) {
                    // Remove any existing messages
                    const existingMessages = modalBody.querySelectorAll('.alert');
                    existingMessages.forEach(msg => msg.remove());

                    modalBody.insertBefore(errorMessage, modalBody.firstChild);
                } else {
                    // Fallback if modal body not found
                    alert('An error occurred while updating the certificate. This may be because the file is too large. Please try again with smaller files (less than 32MB total).');
                }
            })
            .finally(() => {
                // Reset button state
                btn.innerHTML = originalText;
                btn.disabled = false;
            });
        });
    });

    // Initialize selected count on page load
    updateSelectAllCheckbox();

    // Bulk download functionality
    document.getElementById('selectAll').addEventListener('click', function() {
        const checkboxes = document.querySelectorAll('.certificate-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = true;
        });
        updateSelectAllCheckbox();
    });

    document.getElementById('deselectAll').addEventListener('click', function() {
        const checkboxes = document.querySelectorAll('.certificate-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
        });
        updateSelectAllCheckbox();
    });

    document.getElementById('selectAllCheckbox').addEventListener('change', function() {
        const checkboxes = document.querySelectorAll('.certificate-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
    });

    // Update select all checkbox based on individual checkboxes
    function updateSelectAllCheckbox() {
        const checkboxes = document.querySelectorAll('.certificate-checkbox');
        const checkedBoxes = document.querySelectorAll('.certificate-checkbox:checked');
        const selectAllCheckbox = document.getElementById('selectAllCheckbox');
        const selectedCount = document.getElementById('selectedCount');

        // Update selected count
        selectedCount.textContent = `${checkedBoxes.length} selected`;

        if (checkedBoxes.length === 0) {
            selectAllCheckbox.checked = false;
            selectAllCheckbox.indeterminate = false;
        } else if (checkedBoxes.length === checkboxes.length) {
            selectAllCheckbox.checked = true;
            selectAllCheckbox.indeterminate = false;
        } else {
            selectAllCheckbox.checked = false;
            selectAllCheckbox.indeterminate = true;
        }
    }

    // Listen for changes on individual checkboxes
    document.addEventListener('change', function(e) {
        if (e.target.classList.contains('certificate-checkbox')) {
            updateSelectAllCheckbox();
        }
    });

    // Download selected certificates
    document.getElementById('downloadSelected').addEventListener('click', function(e) {
        e.preventDefault();

        const checkedBoxes = document.querySelectorAll('.certificate-checkbox:checked');
        if (checkedBoxes.length === 0) {
            alert('Please select at least one certificate to download.');
            return;
        }

        const certificateIds = Array.from(checkedBoxes).map(checkbox => checkbox.getAttribute('data-cert-id'));

        downloadCertificates(certificateIds, 'selected');
    });

    // Download by category
    document.querySelectorAll('.download-category').forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();

            const category = this.getAttribute('data-category');
            const categoryCheckboxes = document.querySelectorAll(`.certificate-checkbox[data-category="${category}"]`);

            if (categoryCheckboxes.length === 0) {
                alert('No certificates found in this category.');
                return;
            }

            const certificateIds = Array.from(categoryCheckboxes).map(checkbox => checkbox.getAttribute('data-cert-id'));

            downloadCertificates(certificateIds, 'category', category);
        });
    });

    function downloadCertificates(certificateIds, downloadType, category = null) {
        // Create a form to submit the download request
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '{{ route("vessel-certificates.bulk-download") }}';
        form.style.display = 'none';

        // Add CSRF token
        const csrfToken = document.createElement('input');
        csrfToken.type = 'hidden';
        csrfToken.name = '_token';
        csrfToken.value = '{{ csrf_token() }}';
        form.appendChild(csrfToken);

        // Add certificate IDs
        certificateIds.forEach(id => {
            const idInput = document.createElement('input');
            idInput.type = 'hidden';
            idInput.name = 'certificate_ids[]';
            idInput.value = id;
            form.appendChild(idInput);
        });

        // Add download type
        const typeInput = document.createElement('input');
        typeInput.type = 'hidden';
        typeInput.name = 'download_type';
        typeInput.value = downloadType;
        form.appendChild(typeInput);

        // Add vessel name
        const vesselNameInput = document.createElement('input');
        vesselNameInput.type = 'hidden';
        vesselNameInput.name = 'vessel_name';
        vesselNameInput.value = '{{ $vessel->name }}';
        form.appendChild(vesselNameInput);

        // Add category if provided
        if (category) {
            const categoryInput = document.createElement('input');
            categoryInput.type = 'hidden';
            categoryInput.name = 'category';
            categoryInput.value = category;
            form.appendChild(categoryInput);
        }

        // Submit the form
        document.body.appendChild(form);
        form.submit();
        document.body.removeChild(form);
    }

    // Simple attachment deletion (AJAX - no modal closing)
    const simpleDeleteBtns = document.querySelectorAll('.simple-delete-btn');

    simpleDeleteBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            if (confirm('Are you sure you want to delete this attachment?')) {
                const certId = btn.getAttribute('data-cert-id');
                const attachmentIndex = btn.getAttribute('data-index');
                const attachmentPath = btn.getAttribute('data-attachment');

                // Show loading state
                const originalText = btn.innerHTML;
                btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                btn.disabled = true;

                // Send AJAX delete request
                fetch('{{ route("vessel-certificates.remove-attachment") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    body: JSON.stringify({
                        vessel_certificate_id: certId,
                        attachment_index: parseInt(attachmentIndex),
                        attachment_path: attachmentPath
                    })
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        // Remove the attachment item from the display
                        const attachmentItem = btn.closest('.mb-2') || btn.closest('.d-flex');
                        if (attachmentItem) {
                            attachmentItem.remove();
                        }

                        // Show success message in modal
                        showVesselModalAlert('success', 'Attachment deleted successfully!');
                    } else {
                        throw new Error(data.message || 'Failed to delete attachment');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showVesselModalAlert('danger', 'Error deleting attachment: ' + error.message);

                    // Reset button state
                    btn.innerHTML = originalText;
                    btn.disabled = false;
                });
            }
        });
    });

    // Handle attachment deletion (simplified AJAX - no modal closing)
    const deleteAttachmentBtns = document.querySelectorAll('.delete-attachment-btn');

    deleteAttachmentBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            if (confirm('Are you sure you want to delete this attachment?')) {
                const certId = btn.getAttribute('data-cert-id');
                const attachmentIndex = btn.getAttribute('data-index');
                const attachmentPath = btn.getAttribute('data-attachment');

                // Show loading state
                const originalText = btn.innerHTML;
                btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                btn.disabled = true;

                // Send AJAX delete request
                fetch('{{ route("vessel-certificates.remove-attachment") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    body: JSON.stringify({
                        vessel_certificate_id: certId,
                        attachment_index: parseInt(attachmentIndex),
                        attachment_path: attachmentPath
                    })
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        // Remove the attachment item from the display
                        const attachmentItem = btn.closest('.mb-2') || btn.closest('.d-flex');
                        if (attachmentItem) {
                            attachmentItem.remove();
                        }

                        // Show success message in modal
                        showVesselModalAlert('success', 'Attachment deleted successfully!');
                    } else {
                        throw new Error(data.message || 'Failed to delete attachment');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showVesselModalAlert('danger', 'Error deleting attachment: ' + error.message);

                    // Reset button state
                    btn.innerHTML = originalText;
                    btn.disabled = false;
                });
            }
        });
    });
});

// Function to show alerts inside the vessel modal
function showVesselModalAlert(type, message) {
    // Find the modal body (could be in edit modal or any other modal)
    const modalBody = document.querySelector('.modal.show .modal-body');

    if (!modalBody) {
        console.log('No modal found, showing regular alert:', message);
        alert(message);
        return;
    }

    // Remove existing alerts
    const existingAlerts = modalBody.querySelectorAll('.alert');
    existingAlerts.forEach(alert => alert.remove());

    // Create new alert
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show`;
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    // Insert at the top of modal body
    modalBody.insertBefore(alertDiv, modalBody.firstChild);

    // Auto-dismiss after 3 seconds
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 3000);
}
</script>
@endsection
